/*
 * Copyright (C) 2006-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of 
 * the License, or (at your option) any later version. See COPYING.
 *
 * Current QEMU uses a library called "softfloat" to have floating point
 * support on all architectures. FAUmachine however does not really need
 * this, at least not currently, as all supported architectures have the
 * needed floating point operations in hardware.
 * In other words, this file is nothing but a wrapper that maps the library
 * functions that QEMU uses to the native functions.
 */

#include <assert.h>
#include <fenv.h>
#include <inttypes.h>
#include <math.h>
#include <stdio.h>

typedef float float32;
typedef double float64;
typedef long double float80;

typedef struct float_status {
	signed char float_rounding_mode;
	signed char float80_rounding_precision;
} float_status;

static inline __attribute__((__always_inline__)) float32
float32_add(float32 a, float32 b, float_status * status)
{
	return (a + b);
}

static inline __attribute__((__always_inline__)) float64
float64_add(float64 a, float64 b, float_status * status)
{
	return (a + b);
}

static inline __attribute__((__always_inline__)) float32
float32_sub(float32 a, float32 b, float_status * status)
{
	return (a - b);
}

static inline __attribute__((__always_inline__)) float64
float64_sub(float64 a, float64 b, float_status * status)
{
	return (a - b);
}

static inline __attribute__((__always_inline__)) float32
float32_mul(float32 a, float32 b, float_status * status)
{
	return (a * b);
}

static inline __attribute__((__always_inline__)) float64
float64_mul(float64 a, float64 b, float_status * status)
{
	return (a * b);
}

static inline __attribute__((__always_inline__)) float32
float32_div(float32 a, float32 b, float_status * status)
{
	return (a / b);
}

static inline __attribute__((__always_inline__)) float64
float64_div(float64 a, float64 b, float_status * status)
{
	return (a / b);
}

static inline __attribute__((__always_inline__)) float32
float32_sqrt(float32 x, float_status * status)
{
	return (float32)sqrt((float32)x);
}

static inline __attribute__((__always_inline__)) float64
float64_sqrt(float64 x, float_status * status)
{
	return sqrt(x);
}

static inline __attribute__((__always_inline__)) float80
float80_sqrt(float80 x, float_status * status)
{
	return sqrtl(x);
}

static inline __attribute__((__always_inline__)) int32_t
float32_to_int32(float32 x, float_status * status)
{
	long a;

	a = lrint(x);
	if (a != (int32_t) a) {
		a = 1 << 31;
	}

	return a;
}

static inline __attribute__((__always_inline__)) int64_t
float32_to_int64(float32 x, float_status * status)
{
	/* we don't have llrint in our libm */
	return llrintl(x);
}

static inline __attribute__((__always_inline__)) int32_t
float64_to_int32(float64 x, float_status * status)
{
	long a;

	a = lrint(x);
	if (a != (int32_t) a) {
		a = 1 << 31;
	}

	return a;
}

static inline __attribute__((__always_inline__)) int64_t
float64_to_int64(float64 x, float_status * status)
{
	/* we don't have llrint in our libm */
	return llrintl(x);
}

static inline __attribute__((__always_inline__)) int32_t
float80_to_int32(float80 x, float_status * status)
{
	long a;

	a = lrintl(x);
	if (a != (int32_t) a) {
		a = 1 << 31;
	}
	return a;
}

static inline __attribute__((__always_inline__)) int64_t
float80_to_int64(float80 x, float_status * status)
{
	return llrintl(x);
}

static inline __attribute__((__always_inline__)) int32_t
float32_to_int32_round_to_zero(float32 x, float_status * status)
{
	return (int32_t)x;
}

static inline __attribute__((__always_inline__)) int64_t
float32_to_int64_round_to_zero(float32 x, float_status * status)
{
	return (int64_t)x;
}

static inline __attribute__((__always_inline__)) int32_t
float64_to_int32_round_to_zero(float64 x, float_status * status)
{
	return (int32_t)x;
}

static inline __attribute__((__always_inline__)) int64_t
float64_to_int64_round_to_zero(float64 x, float_status * status)
{
	return (int64_t)x;
}

static inline __attribute__((__always_inline__)) float64
float32_to_float64(float32 x, float_status * status)
{
	return (float64)x;
}

static inline __attribute__((__always_inline__)) float32
float64_to_float32(float64 x, float_status * status)
{
	return (float32)x;
}

static inline __attribute__((__always_inline__)) float32
int32_to_float32(int32_t x, float_status * status)
{
	return (float32)x;
}

static inline __attribute__((__always_inline__)) float64
int32_to_float64(int32_t x, float_status * status)
{
	return (float64)x;
}

static inline __attribute__((__always_inline__)) float32
int64_to_float32(int64_t x, float_status * status)
{
	return (float32)x;
}

static inline __attribute__((__always_inline__)) float64
int64_to_float64(int64_t x, float_status * status)
{
	return (float64)x;
}

static inline __attribute__((__always_inline__)) char
float32_eq(float32 a, float32 b, float_status * status)
{
	return (a == b);
}

static inline __attribute__((__always_inline__)) char
float64_eq(float64 a, float64 b, float_status * status)
{
	return (a == b);
}

static inline __attribute__((__always_inline__)) char
float32_lt(float32 a, float32 b, float_status * status)
{
	return (a < b);
}

static inline __attribute__((__always_inline__)) char
float64_lt(float64 a, float64 b, float_status * status)
{
	return (a < b);
}

static inline __attribute__((__always_inline__)) char
float32_le(float32 a, float32 b, float_status * status)
{
	return (a <= b);
}

static inline __attribute__((__always_inline__)) char
float64_le(float64 a, float64 b, float_status * status)
{
	return (a <= b);
}

/* Supposed difference between compare and compare_quiet:
 * compare can generate an exception when one of the numbers is NaN */
static inline __attribute__((__always_inline__)) char
float32_compare(float32 a, float32 b, float_status * status)
{
	if        (a <  b) {
		return -1;
	} else if (a == b) {
		return  0;
	} else if (a >  b) {
		return  1;
	}
	return  2;
}

static inline __attribute__((__always_inline__)) char
float64_compare(float64 a, float64 b, float_status * status)
{
	if        (a <  b) {
		return -1;
	} else if (a == b) {
		return  0;
	} else if (a >  b) {
		return  1;
	}
	return  2;
}

static inline __attribute__((__always_inline__)) char
float80_compare(float80 a, float80 b, float_status * status)
{
	if        (a <  b) {
		return -1;
	} else if (a == b) {
		return  0;
	} else if (a >  b) {
		return  1;
	}
	return  2;
}

static inline __attribute__((__always_inline__)) char
float32_compare_quiet(float32 a, float32 b, float_status * status)
{
	if        (isless(a, b)) {
		return -1;
	} else if (a == b) {
		return  0;
	} else if (isgreater(a, b)) {
		return  1;
	}
	return  2;
}

static inline __attribute__((__always_inline__)) char
float64_compare_quiet(float64 a, float64 b, float_status * status)
{
	if        (isless(a, b)) {
		return -1;
	} else if (a == b) {
		return  0;
	} else if (isgreater(a, b)) {
		return  1;
	}
	return  2;
}

static inline __attribute__((__always_inline__)) char
float80_compare_quiet(float80 a, float80 b, float_status * status)
{
	if        (isless(a, b)) {
		return -1;
	} else if (a == b) {
		return  0;
	} else if (isgreater(a, b)) {
		return  1;
	}
	return  2;
}

static inline __attribute__((__always_inline__)) char
float32_unordered(float32 a, float32 b, float_status * status)
{
	return isunordered(a, b);
}

static inline __attribute__((__always_inline__)) char
float64_unordered(float64 a, float64 b, float_status * status)
{
	return isunordered(a, b);
}

static inline __attribute__((__always_inline__)) void
set_float_rounding_mode(int val, float_status * status)
{
	status->float_rounding_mode = val;
	fesetround(val);
}

static inline __attribute__((__always_inline__)) float80
float80_abs(float80 x)
{
	return fabsl(x);
}

static inline __attribute__((__always_inline__)) float80
float80_chs(float80 x)
{
	return -x;
}
