/*
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* This will turn on packet command debug output. */
#define DEBUGPCOM	0

/*
 * Config Options
 */
#define CD_AUDIO_SUPPORT		0
#define CD_WRITER_SUPPORT		1
#define CD_CHANGER_SUPPORT		0

/* CD_WRITER_SUPPORT */
#define CD_WRITER_RW_SUPPORT		0
#define CD_WRITER_DAO_SUPPORT		1

#define CD_FORMAT_UNIT_SUPPORT		0
#define CD_READ_BUFFER_CAPACITY_SUPPORT	0
#define CD_READ_MASTER_CUE_SUPPORT	0
#define CD_SEND_CUE_SHEET_SUPPORT	0
#define CD_SEND_OPC_INFORMATION_SUPPORT	0

#define CD_PLAY_CD_SUPPORT		0
#define CD_READ_CD_SUPPORT		0
#define CD_SCAN_CD_SUPPORT		0

#include "config.h"

#include <assert.h>
#include <fcntl.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "glue.h"

#include "std-ide.h"

struct cpssp {
	/* Config */
	char name[1024];

	/* Ports */
	struct sig_magneto_optical *port_media;
	struct sig_ide_bus *port_ide;
	struct sig_string *port_change;

	/* State */
	struct {
		unsigned int unit;

		uint8_t error;		/* error reg (ro) */
		uint8_t features;	/* features reg (wo) */
		uint8_t nsector;	/* irq_status/sector count reg */
		uint8_t sector;		/* lba low reg (r/w) */
		uint16_t cyl;		/* byte count reg (r/w) */
		uint8_t select;		/* select reg (r/w) */
		uint8_t command;	/* command reg (wo) */
		uint8_t control;	/* device control reg (ro) */
	} common;

#define STATE

#define ATAPI		1
#define NAME		cdrom
#define NAME_(x)	cdrom_ ## x
#define SNAME		"cdrom"
#include "arch_scsi_gen_cdrom.c"
#undef SNAME
#undef NAME_
#undef NAME
#undef ATAPI

#define NAME		atapi
#define NAME_(x)	atapi_ ## x
#define SNAME		"atapi"
#include "arch_gen_atapi.c"
#undef SNAME
#undef NAME_
#undef NAME

#define NAME		iface
#define NAME_(x)	iface_ ## x
#define SNAME		"iface"
#include "arch_pata_ata.c"
#undef SNAME
#undef NAME_
#undef NAME

#undef STATE
};

/*forward*/ static void
cdrom_atn_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static int
cdrom_phase_select(struct cpssp *cpssp, uint32_t id);
/*forward*/ static int
cdrom_recv(struct cpssp *cpssp, uint8_t *buf, unsigned int bufsize);
/*forward*/ static int
cdrom_send(struct cpssp *cpssp, const uint8_t *buf, unsigned int bufsize);

/*forward*/ static void
atapi_phase_free(struct cpssp *cpssp);
/*forward*/ static void
atapi_phase_cmd(struct cpssp *cpssp);
/*forward*/ static void
atapi_phase_status(struct cpssp *cpssp);
/*forward*/ static void
atapi_phase_data_in(struct cpssp *cpssp);
/*forward*/ static void
atapi_phase_data_out(struct cpssp *cpssp);
/*forward*/ static void
atapi_phase_msg_in(struct cpssp *cpssp);
/*forward*/ static void
atapi_phase_msg_out(struct cpssp *cpssp);
/*forward*/ static void
atapi_should_recv(struct cpssp *cpssp, unsigned long count);
/*forward*/ static void
atapi_should_send(struct cpssp *cpssp, unsigned long count);
/*forward*/ static void
atapi_command(struct cpssp *cpssp);
/*forward*/ static void
atapi_soft_reset(struct cpssp *cpssp);
/*forward*/ static void
atapi_data_in(struct cpssp *cpssp, uint16_t *valp);
/*forward*/ static void
atapi_data_out(struct cpssp *cpssp, uint16_t val);

/*forward*/ static void
iface_dma_in(struct cpssp *cpssp, unsigned int count);
/*forward*/ static void
iface_dma_out(struct cpssp *cpssp, unsigned int count);
/*forward*/ static void
iface_pio_in(struct cpssp *cpssp, unsigned int count, int irq);
/*forward*/ static void
iface_pio_out(struct cpssp *cpssp, unsigned int count, int irq);
/*forward*/ static void
iface_done_io(struct cpssp *cpssp);
/*forward*/ static void
iface_done_cmd(struct cpssp *cpssp, int irq);

static void
cdrom_want_recv(struct cpssp *cpssp, unsigned int count)
{
	atapi_should_send(cpssp, count);
}

static void
cdrom_want_send(struct cpssp *cpssp, unsigned int count)
{
	atapi_should_recv(cpssp, count);
}

static void
cdrom_phase_free(struct cpssp *cpssp)
{
	atapi_phase_free(cpssp);
}

static void
cdrom_phase_cmd(struct cpssp *cpssp)
{
	atapi_phase_cmd(cpssp);
}

static void
cdrom_phase_status(struct cpssp *cpssp)
{
	atapi_phase_status(cpssp);
}

static void
cdrom_phase_data_in(struct cpssp *cpssp)
{
	atapi_phase_data_in(cpssp);
}

static void
cdrom_phase_data_out(struct cpssp *cpssp)
{
	atapi_phase_data_out(cpssp);
}

static void
cdrom_phase_msg_in(struct cpssp *cpssp)
{
	atapi_phase_msg_in(cpssp);
}

static void
cdrom_phase_msg_out(struct cpssp *cpssp)
{
	atapi_phase_msg_out(cpssp);
}

static void
cdrom_load_unload(struct cpssp *cpssp, int unload)
{
	if (! unload) {
		/* Load Disk */
		/* Nothing to do... */
	} else {
		/* Unload Disk */
		sig_string_set(cpssp->port_change, cpssp, "");
	}
}

static void
atapi_dma_in(struct cpssp *cpssp, unsigned int count)
{
	iface_dma_in(cpssp, count);
}

static void
atapi_dma_out(struct cpssp *cpssp, unsigned int count)
{
	iface_dma_out(cpssp, count);
}

static void
atapi_pio_in(struct cpssp *cpssp, unsigned int count, int irq)
{
	iface_pio_in(cpssp, count, irq);
}

static void
atapi_pio_out(struct cpssp *cpssp, unsigned int count, int irq)
{
	iface_pio_out(cpssp, count, irq);
}

static void
atapi_done_io(struct cpssp *cpssp)
{
	iface_done_io(cpssp);
}

static void
atapi_done_cmd(struct cpssp *cpssp, int irq)
{
	iface_done_cmd(cpssp, irq);
}

static void
atapi_phase_select(struct cpssp *cpssp)
{
	cdrom_atn_set(cpssp, 0);
	(void) cdrom_phase_select(cpssp, 0);
}

static int
atapi_send(struct cpssp *cpssp, const void *buf, unsigned int bufsize)
{
	return cdrom_send(cpssp, (const uint8_t *) buf, bufsize);
}

static int
atapi_recv(struct cpssp *cpssp, void *buf, unsigned int bufsize)
{
	return cdrom_recv(cpssp, (uint8_t *) buf, bufsize);
}

static void
iface_dmarq_out_set(struct cpssp *cpssp, unsigned int val)
{
	sig_ide_bus_dmarq_set(cpssp->port_ide, val);
}

static void
iface_irqrq_out_set(struct cpssp *cpssp, unsigned int val)
{
	sig_ide_bus_irq(cpssp->port_ide, cpssp, val);
}

static void
iface_command(struct cpssp *cpssp)
{
	atapi_command(cpssp);
}

static void
iface_soft_reset(struct cpssp *cpssp)
{
	atapi_soft_reset(cpssp);
}

static void
iface_data_in(struct cpssp *cpssp, uint16_t *valp)
{
	atapi_data_in(cpssp, valp);
}

static void
iface_data_out(struct cpssp *cpssp, uint16_t val)
{
	atapi_data_out(cpssp, val);
}

#define BEHAVIOR

#define ATAPI		1
#define NAME		cdrom
#define NAME_(x)	cdrom_ ## x
#define SNAME		"cdrom"
#include "arch_scsi_gen_cdrom.c"
#undef SNAME
#undef NAME_
#undef NAME
#undef ATAPI

#define NAME		atapi
#define NAME_(x)	atapi_ ## x
#define SNAME		"atapi"
#include "arch_gen_atapi.c"
#undef SNAME
#undef NAME_
#undef NAME

#define DEBUG			0
#define PACKET_INTERFACE	1
#define NAME		iface
#define NAME_(x)	iface_ ## x
#define SNAME		"iface"
#include "arch_pata_ata.c"
#undef SNAME
#undef NAME_
#undef NAME
#undef PACKET_INTERFACE
#undef DEBUG

#undef BEHAVIOR

static void
COMP_(power_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cdrom_power_set(cpssp, val);
	atapi_power_set(cpssp, val);
	iface_power_set(cpssp, val);
}

static void
COMP_(reset)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cdrom_reset_set(cpssp, val);
	atapi_reset_set(cpssp, val);
	iface_reset_set(cpssp, val);
}

static int
COMP_(inw)(void *_cpssp, unsigned short port, uint16_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	return iface_inw(cpssp, port, valp);
}

static void
COMP_(outw)(void *_cpssp, unsigned short port, uint16_t value)
{
	struct cpssp *cpssp = _cpssp;

	iface_outw(cpssp, port, value);
}

static int
COMP_(dmainw)(void *_cpssp, uint16_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	return iface_dmainw(cpssp, valp);
}

static void
COMP_(dmaoutw)(void *_cpssp, uint16_t value)
{
	struct cpssp *cpssp = _cpssp;

	iface_dmaoutw(cpssp, value);
}

static void
COMP_(change)(void *_cpssp, const char *path)
{
	struct cpssp *cpssp = _cpssp;

	cdrom_change(cpssp, path);
}

void *
COMP_(create)(
	const char *name,
	const char *unit_str,
	struct sig_manage *port_manage,
	struct sig_molex_ide_power_conn *port_power,
	struct sig_ide_bus *port_ide,
	struct sig_magneto_optical *port_media,
	struct sig_string *port_change
)
{
	static const struct sig_std_logic_funcs power_funcs = {
		.boolean_or_set = COMP_(power_set),
	};
	static const struct sig_ide_bus_funcs funcs = {
		.reset = COMP_(reset),
		.inw = COMP_(inw),
		.outw = COMP_(outw),
		.dmainw = COMP_(dmainw),
		.dmaoutw = COMP_(dmaoutw),
	};
	static const struct sig_string_funcs change_funcs = {
		.set = COMP_(change),
	};
	struct cpssp *cpssp;
	char path[1024];

	if (! unit_str) unit_str = "0";

	system_name_push(name);

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	assert(strlen(system_path()) < sizeof(cpssp->name));
	strcpy(cpssp->name, system_path());

	cpssp->common.unit = strtoul(unit_str, NULL, 0);

	assert(strlen(system_path()) + strlen(".media") + 1 <= sizeof(path));
	sprintf(path, "%s.media", system_path());

	cdrom_create(cpssp, path, 0);

	atapi_create(cpssp);

	iface_create(cpssp);

	sig_std_logic_connect_in(port_power->_plus_5V, cpssp, &power_funcs);

	cpssp->port_ide = port_ide;
	sig_ide_bus_connect(port_ide, cpssp, &funcs);

	cpssp->port_media = port_media;

	cpssp->port_change = port_change;
	sig_string_connect(port_change, cpssp, &change_funcs);

	system_name_pop();

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	cdrom_destroy(cpssp);
	atapi_destroy(cpssp);
	iface_destroy(cpssp);

	shm_free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	cdrom_suspend(cpssp, fp);
	atapi_suspend(cpssp, fp);
	iface_suspend(cpssp, fp);
}

void
COMP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	cdrom_resume(cpssp, fp);
	atapi_resume(cpssp, fp);
	iface_resume(cpssp, fp);
}
