// SPDX-FileCopyrightText: 2018 - 2023 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include "vpncontrollernm.h"

#include <QDBusAbstractInterface>

#include <NetworkManagerQt/Manager>
#include <NetworkManagerQt/Settings>
#include <NetworkManagerQt/ConnectionSettings>
#include <NetworkManagerQt/VpnSetting>

#define NETWORKSERVICE "com.deepin.system.Network"
#define NETWORKPATH "/com/deepin/system/Network"
#define NETWORKINTERFACE "com.deepin.system.Network"
#define NETWORKPROPERTIESINTERFACE "org.freedesktop.DBus.Properties"
#define VPNENABLED "VpnEnabled"

using namespace dde::network;

VPNController_NM::VPNController_NM(QObject *parent)
    : VPNController (parent)
{
    initMember();
    initConnections();
}

VPNController_NM::~VPNController_NM()
{
    for (VPNItem *vpnItem : m_items)
        delete vpnItem;
}

void VPNController_NM::initMember()
{
    QList<VPNItem *> newItems;
    NetworkManager::Connection::List connections = NetworkManager::listConnections();
    for (NetworkManager::Connection::Ptr connection : connections) {
        VPNItem *newItem = addVpnConnection(connection);
        if (newItem)
            newItems << newItem;
    }

    sortVPNItems();

    Q_EMIT itemAdded(newItems);

    QMetaObject::invokeMethod(this, &VPNController_NM::onActiveConnectionsChanged);
}

void VPNController_NM::initConnections()
{
    connect(NetworkManager::settingsNotifier(), &NetworkManager::SettingsNotifier::connectionAdded, this, &VPNController_NM::onConnectionAdded);
    connect(NetworkManager::settingsNotifier(), &NetworkManager::SettingsNotifier::connectionRemoved, this, &VPNController_NM::onConnectionRemoved);
    connect(NetworkManager::notifier(), &NetworkManager::Notifier::activeConnectionsChanged, this, &VPNController_NM::onActiveConnectionsChanged);
    QDBusConnection::systemBus().connect(NETWORKSERVICE, NETWORKPATH, NETWORKPROPERTIESINTERFACE,
                                         "PropertiesChanged", this, SLOT(onPropertiesChanged(const QString &, const QVariantMap &)));
}

VPNItem *VPNController_NM::addVpnConnection(const NetworkManager::Connection::Ptr &connection)
{
    if (connection->settings()->connectionType() != NetworkManager::ConnectionSettings::ConnectionType::Vpn)
        return nullptr;

    auto createJson = [](const NetworkManager::Connection::Ptr &connection) {
        QJsonObject json;
        json.insert("Path", connection->path());
        json.insert("Uuid", connection->uuid());
        json.insert("Id", connection->settings()->id());
        json.insert("IfcName", connection->settings()->interfaceName());
        json.insert("HwAddress", QString());
        json.insert("ClonedAddress", QString());
        json.insert("Ssid", QString());
        json.insert("Hidden", false);
        return json;
    };
    // 从列表中查找是否存在path相同的连接
    QList<VPNItem *>::iterator itVpn = std::find_if(m_items.begin(), m_items.end(), [ connection ](VPNItem *item) { return item->connection()->path() == connection->path(); });
    if (itVpn != m_items.end()) {
        VPNItem *vpnItem = *itVpn;
        vpnItem->setConnection(createJson(connection));
        return nullptr;
    }
    // 如果不存在相同的连接，则新建一个连接
    VPNItem *vpnItem = new VPNItem;
    vpnItem->setConnection(createJson(connection));
    vpnItem->updateTimeStamp(connection->settings()->timestamp());
    m_items << vpnItem;
    m_vpnConnectionsMap[vpnItem] = connection;
    connect(connection.data(), &NetworkManager::Connection::updated, this, [ connection, vpnItem, createJson, this ] {
        // 更新数据
        vpnItem->setConnection(createJson(connection));
        Q_EMIT activeConnectionChanged();
    });

    return vpnItem;
}

void VPNController_NM::sortVPNItems()
{
    std::sort(m_items.begin(), m_items.end(), [](VPNItem *item1, VPNItem *item2) {
        return item1->connection()->id() < item2->connection()->id();
    });
}

VPNItem *VPNController_NM::findFirstAutoConnectItem() const
{
    if (m_items.size() == 0)
        return nullptr;

    // 排序，先将最后连接的放到最前面,从列表中查找可以自动连接的
    QMap<QString, VPNItem *> vpnItemMap;
    for (VPNItem *vpnItem : m_items)
        vpnItemMap[vpnItem->connection()->path()] = vpnItem;
    QList<QPair<VPNItem *, QDateTime>> vpnItems;
    NetworkManager::Connection::List connections = NetworkManager::listConnections();
    for (NetworkManager::Connection::Ptr connection : connections) {
        // 只查找VPN的连接的类型
        if (connection->settings()->connectionType() != NetworkManager::ConnectionSettings::ConnectionType::Vpn)
            continue;

        // 过滤非自动连接的
        if (!vpnItemMap.contains(connection->path()) || !connection->settings()->autoconnect())
            continue;

        vpnItems << qMakePair(vpnItemMap[connection->path()], connection->settings()->timestamp());
    }
    // 如果没有自动连接的VPN，则返回false
    if (vpnItems.size() == 0)
        return nullptr;

    std::sort(vpnItems.begin(), vpnItems.end(), [ = ](const QPair<VPNItem *, QDateTime> &item1, const QPair<VPNItem *, QDateTime> &item2) {
        if (!item1.second.isValid() && !item2.second.isValid())
            return item1.first->connection()->id() > item2.first->connection()->id();
        if (!item1.second.isValid() && item2.second.isValid())
            return false;
        if (item1.second.isValid() && !item2.second.isValid())
            return true;

        return item1.second > item2.second;
    });

    return vpnItems.first().first;
}

NetworkManager::ActiveConnection::Ptr VPNController_NM::findActiveConnection() const
{
    NetworkManager::ActiveConnection::List activeConnections = NetworkManager::activeConnections();
    NetworkManager::ActiveConnection::List::iterator itActiveConnection = std::find_if(activeConnections.begin(), activeConnections.end(), [](NetworkManager::ActiveConnection::Ptr activeConnection) {
        return (activeConnection->connection()->settings()->connectionType() == NetworkManager::ConnectionSettings::ConnectionType::Vpn);
    });

    if (itActiveConnection == activeConnections.end())
        return NetworkManager::ActiveConnection::Ptr();

    return *itActiveConnection;
}

NetworkManager::Connection::Ptr VPNController_NM::findConnectionByVPNItem(VPNItem *vpnItem) const
{
    NetworkManager::Connection::List connections = NetworkManager::listConnections();
    NetworkManager::Connection::List::iterator itVpn = std::find_if(connections.begin(), connections.end(), [ vpnItem, this ](NetworkManager::Connection::Ptr connection) {
        if (connection->settings()->connectionType() != NetworkManager::ConnectionSettings::ConnectionType::Vpn)
            return false;

        return vpnItem->connection()->path() == connection->path();
    });

    if (itVpn == connections.end())
        return NetworkManager::Connection::Ptr();

    return *itVpn;
}

void VPNController_NM::onConnectionAdded(const QString &path)
{
    qInfo() << "new vpn connection:" << path;
    NetworkManager::Connection::List connections = NetworkManager::listConnections();
    NetworkManager::Connection::List::iterator itConnection = std::find_if(connections.begin(), connections.end(), [ path ](NetworkManager::Connection::Ptr connection) {
        return connection->path() == path;
    });

    if (itConnection == connections.end())
        return;

    VPNItem *item = addVpnConnection(*itConnection);
    if (item) {
        sortVPNItems();
        Q_EMIT itemAdded({ item });
    }
}

void VPNController_NM::onConnectionRemoved(const QString &path)
{
    qInfo() << "remove connection:" << path;
    for (VPNItem *item : m_items) {
        if (item->connection()->path() != path)
            continue;

        m_items.removeAll(item);
        m_vpnConnectionsMap.remove(item);
        Q_EMIT itemRemoved({ item });
        delete item;
        break;
    }
}

void VPNController_NM::onActiveConnectionsChanged()
{
    // 活动连接发生变化
    NetworkManager::ActiveConnection::Ptr activeConnection = findActiveConnection();
    if (activeConnection.isNull() || activeConnection->connection()->settings()->connectionType() != NetworkManager::ConnectionSettings::ConnectionType::Vpn)
        return;

    PRINT_INFO_MESSAGE << "active connection changed:" << activeConnection->path();
    // 先获取当前的VPN类型
    const QString &activeServiceType = activeConnection->connection()->settings()->setting(NetworkManager::Setting::SettingType::Vpn).
            staticCast<NetworkManager::VpnSetting>()->serviceType();

    // 将当前所有同类型的其他的VPN状态置未连接，获取所有连接的状态，遍历所有的VPN连接，只处理VPN类型和当前类型相同的连接
    VPNItem *activeItem = nullptr;
    for (VPNItem *item : m_items) {
        if (m_vpnConnectionsMap.contains(item)) {
            // 不同类型的VPN跳过即可，无需设置其状态为断开
            NetworkManager::Connection::Ptr connection = m_vpnConnectionsMap[item];
            const QString &serviceType = connection->settings()->setting(NetworkManager::Setting::SettingType::Vpn).
                    staticCast<NetworkManager::VpnSetting>()->serviceType();
            if (serviceType != activeServiceType)
                continue;
        }

        item->setActiveConnection(QString());
        if (item->connection()->path() == activeConnection->connection()->path()) {
            activeItem = item;
        } else {
            item->setConnectionStatus(ConnectionStatus::Deactivated);
        }
    }

    if (activeItem) {
        connect(activeConnection.data(), &NetworkManager::ActiveConnection::stateChanged, this, [ this, activeConnection ](NetworkManager::ActiveConnection::State state) {
            QList<VPNItem *>::iterator itVpnItem = std::find_if(m_items.begin(), m_items.end(), [ activeConnection ](VPNItem *item) { return item->connection()->path() == activeConnection->connection()->path(); });
            if (itVpnItem == m_items.end())
                return;

            VPNItem *activeItem = *itVpnItem;
            ConnectionStatus status = convertStateFromNetworkManager(state);
            activeItem->setConnectionStatus(status);
            if (status == ConnectionStatus::Activated) {
                activeConnection->connection()->settings()->setTimestamp(QDateTime::currentDateTime());
                activeItem->updateTimeStamp(activeConnection->connection()->settings()->timestamp());
                activeItem->setActiveConnection(activeConnection->path());
            }
            Q_EMIT activeConnectionChanged();
        }, Qt::UniqueConnection);

        ConnectionStatus status = convertStateFromNetworkManager(activeConnection->state());
        activeItem->setConnectionStatus(status);
        if (status == ConnectionStatus::Activated) {
            activeItem->updateTimeStamp(activeConnection->connection()->settings()->timestamp());
            activeItem->setActiveConnection(activeConnection->path());
        }

        // 通知界面刷新状态
        Q_EMIT activeConnectionChanged();
    }
}

void VPNController_NM::onPropertiesChanged(const QString &interfaceName, const QVariantMap &changedProperties)
{
    if (interfaceName != NETWORKINTERFACE)
        return;

    if (!changedProperties.contains(VPNENABLED))
        return;

    Q_EMIT enableChanged(changedProperties.value(VPNENABLED).toBool());
}

void VPNController_NM::setEnabled(const bool enabled)
{
    // 调用com.deepin.system.Network的相关接口来设置VPN
    QDBusInterface dbusInter(NETWORKSERVICE, NETWORKPATH, NETWORKINTERFACE, QDBusConnection::systemBus());
    dbusInter.setProperty(VPNENABLED, enabled);
    // 如果开启VPN，则让其自动连接
    if (!enabled)
        return;

    // 查找最近一次自动连接的VPN
    VPNItem *firstConnectionItem = findFirstAutoConnectItem();
    if (firstConnectionItem)
        connectItem(firstConnectionItem);
}

bool VPNController_NM::enabled() const
{
    QDBusInterface dbusInter(NETWORKSERVICE, NETWORKPATH, NETWORKINTERFACE, QDBusConnection::systemBus());
    return dbusInter.property(VPNENABLED).toBool();
}

void VPNController_NM::connectItem(VPNItem *item)
{
    if (!item)
        return;

    PRINT_INFO_MESSAGE << QString("connect to Vpn:%1, path:%2").arg(item->connection()->id()).arg(item->connection()->path());
    // 查看当前已经连接的VPN
    NetworkManager::Connection::Ptr connection = findConnectionByVPNItem(item);
    if (connection.isNull()) {
        NetworkManager::activateConnection(item->connection()->path(), "/", "/");
        return;
    }

    const QString &serviceType = connection->settings()->setting(NetworkManager::Setting::SettingType::Vpn).
            staticCast<NetworkManager::VpnSetting>()->serviceType();
    // 如果是相同类型的连接，则只允许连接一个，在连接之前需要先把之前的已经连接的同类型的连接断开
    NetworkManager::ActiveConnection::Ptr activeConnection = findActiveConnection();
    if (!activeConnection.isNull() && connection != activeConnection->connection() &&
            activeConnection->connection()->settings()->setting(NetworkManager::Setting::SettingType::Vpn).
            staticCast<NetworkManager::VpnSetting>()->serviceType() == serviceType) {
        // 如果当前活动的连接和要连接的VPN类型相同，泽断开当前的连接，然后才能连接下一个连接
        PRINT_INFO_MESSAGE << QString("deactivate Connection Type: %1,id:%2, path: %3").arg(serviceType)
                           .arg(activeConnection->connection()->settings()->id()).arg(activeConnection->path());

        QDBusPendingCallWatcher *w = new QDBusPendingCallWatcher(NetworkManager::deactivateConnection(activeConnection->path()), this);
        connect(w, &QDBusPendingCallWatcher::finished, w, &QDBusPendingCallWatcher::deleteLater);
        connect(w, &QDBusPendingCallWatcher::finished, this, [ item ] {
            // 需要等断开连接的操作执行完毕后，才能执行连接操作，否则会出现连接失败的情况
            NetworkManager::activateConnection(item->connection()->path(), "/", "/");
        });
    } else {
        NetworkManager::activateConnection(item->connection()->path(), "/", "/");
    }
}

void VPNController_NM::connectItem(const QString &uuid)
{
    QList<VPNItem *>::iterator itVpn = std::find_if(m_items.begin(), m_items.end(), [ uuid ](VPNItem *item) { return item->connection()->uuid() == uuid; });
    if (itVpn != m_items.end()) {
        PRINT_INFO_MESSAGE << "connect connection, uuid:" << uuid;
        connectItem(*itVpn);
    } else {
        PRINT_INFO_MESSAGE << "count found vpn item, uuid:" << uuid;
    }
}

void VPNController_NM::disconnectItem()
{
    NetworkManager::ActiveConnection::Ptr activeConnection = findActiveConnection();
    if (!activeConnection.isNull()) {
        PRINT_INFO_MESSAGE << "disconnect vpn item:" << activeConnection->path();
        NetworkManager::deactivateConnection(activeConnection->path());
    }
}

QList<VPNItem *> VPNController_NM::items() const
{
    return m_items;
}
