// SPDX-FileCopyrightText: 2022 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include "connectivitychecker.h"

#include <DConfig>

#include <QEventLoop>
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QTimer>

//当没有进行配置的时候, 则访问我们官网
static const QStringList CheckUrls {
    "https://www.uniontech.com",
};

#define TIMEOUT (5 * 1000)         // 5s超时

using namespace dde::network;

ConnectivityChecker::ConnectivityChecker(QObject *parent)
    : QObject(parent)
    , m_dConfig(Dtk::Core::DConfig::create("org.deepin.dde.network", "org.deepin.dde.network", QString(), this))
    , m_isRunning(false)
{
    QStringList urls;
    if (m_dConfig->keyList().contains("NetworkCheckerUrls")) {
        urls = m_dConfig->value("NetworkCheckerUrls").toStringList();
        connect(m_dConfig, &Dtk::Core::DConfig::valueChanged, this, &ConnectivityChecker::onConfigChanged);
    }

    updateUrls(urls);
}

void ConnectivityChecker::startCheck()
{
    if (m_isRunning) {
        Q_EMIT runFinished();
        return;
    }

    m_isRunning = true;
    QNetworkAccessManager networkAccessManager;
    for (const QString &url : m_checkUrls) {
        QScopedPointer<QNetworkReply> reply(networkAccessManager.head(QNetworkRequest(QUrl(url))));
        qDebug() << "Check connectivity using url:" << url;

        // Do not use waitForReadyRead to block thread,
        // the QNetworkReply is not implement this virtual method
        // and it will just return false immediately
        // reply->waitForReadyRead(-1);

        // Blocking, about 5 second to timeout
        QTimer timer;
        timer.setSingleShot(true);
        QEventLoop synchronous;
        connect(&timer, &QTimer::timeout, &synchronous, &QEventLoop::quit);
        connect(&networkAccessManager, &QNetworkAccessManager::finished, &synchronous, &QEventLoop::quit);
        timer.start(TIMEOUT);
        synchronous.exec();

        reply->close();
        // 网络状态码中, 大于等于200, 小于等于206的都是网络正常
        if (timer.isActive()) {
            timer.stop();
            if (reply->error() == QNetworkReply::NoError &&
                    (reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt() >= 200 &&
                    reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt() <= 206)) {
                qDebug() << "Connected to url:" << url;
                Q_EMIT checkFinished(true);
                m_isRunning = false;
                Q_EMIT runFinished();
                return;
            }
        } else {
            qDebug() << "Timeout";
        }
    }

    Q_EMIT checkFinished(false);
    m_isRunning = false;
    Q_EMIT runFinished();
}

void ConnectivityChecker::updateUrls(const QStringList &urls)
{
    m_checkUrls = urls;
    if (m_checkUrls.isEmpty())
        m_checkUrls = CheckUrls;
}

void ConnectivityChecker::onConfigChanged(const QString &key)
{
    if (key != QString("NetworkCheckerUrls"))
        return;

    updateUrls(m_dConfig->value("NetworkCheckerUrls").toStringList());
}
