// SPDX-FileCopyrightText: 2018 - 2022 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include "thememanager.h"
#include "utils.h"

#include <DApplication>
#include <DPlatformTheme>
#include <DLineEdit>

DGUI_USE_NAMESPACE

ThemeManager::ThemeManager(QObject *parent)
    : QObject(parent)
    , m_themeType(UnknownType)
{
    appThemeTypeChanged(Dtk::Gui::DGuiApplicationHelper::instance()->themeType());
    // 主题发生变化触发的信号
    connect(Dtk::Gui::DGuiApplicationHelper::instance(), &Dtk::Gui::DGuiApplicationHelper::themeTypeChanged, this, &ThemeManager::appThemeTypeChanged);
}

ThemeManager::~ThemeManager()
{
}

QString ThemeManager::getIcon(QString path)
{
    if (path.startsWith("common")) {
        return ":/" + path;
    }
    switch (m_themeType) {
    case GreeterType:
    case LockType:
    case DarkType:
        return ":/dark/" + path;
    default:
        return ":/light/" + path;
    }
}

void ThemeManager::setThemeType(ThemeType type)
{
    if (type != m_themeType) {
        m_themeType = type;
        for (auto it = m_editList.begin(); it != m_editList.end();) {
            if (it->isNull()) {
                it = m_editList.erase(it);
            } else {
                setInputStyle(it->data());
                it++;
            }
        }
        emit themeChanged(m_themeType);
    }
}

void ThemeManager::updateInputStyle(Dtk::Widget::DLineEdit *inputEdit)
{
    m_editList.append(QPointer<Dtk::Widget::DLineEdit>(inputEdit));
    setInputStyle(inputEdit);
}

void ThemeManager::setInputStyle(Dtk::Widget::DLineEdit *inputEdit)
{
    QColor color(Qt::white);
    switch (m_themeType) {
    case GreeterType:
    case LockType: {
        color.setAlphaF(0.4);
    } break;
    case DarkType: {
        color = Qt::black;
        color.setAlphaF(0.5);
    } break;
    case LightType: {
        color.setAlphaF(0.5);
    } break;
    default:
        break;
    }
    QPalette editPalette = inputEdit->palette();
    editPalette.setColor(QPalette::Button, color);
    inputEdit->setPalette(editPalette);
}

void ThemeManager::appThemeTypeChanged(Dtk::Gui::DGuiApplicationHelper::ColorType themeType)
{
    if (m_themeType >= UnknownType) {
        switch (themeType) {
        case Dtk::Gui::DGuiApplicationHelper::ColorType::DarkType:
            setThemeType(DarkType);
            break;
        case Dtk::Gui::DGuiApplicationHelper::ColorType::LightType:
            setThemeType(LightType);
            break;
        default:
            break;
        }
    }
}

QColor ThemeManager::backgroundColor()
{
    QColor color(m_themeType == DarkType ? Qt::black : Qt::white);
    color.setAlphaF(Utils::isWaylandEnvironment() ? 0.5 : 0.05);
    return color;
}

QColor ThemeManager::lineColor()
{
    QColor color(m_themeType == DarkType ? Qt::white : Qt::black);
    color.setAlphaF(0.1);
    return color;
}

QColor ThemeManager::itemBackgroundColor()
{
    QColor color(m_themeType == LightType ? Qt::black : Qt::white);
    color.setAlphaF(0.1);
    return color;
}
