#pragma once

#include <QString>
#include <QList>
#include <QJsonObject>
#include <QJsonArray>

const int MAIN_WINDOW_WIDTH = 860;
const int MAIN_WINDOW_HEIGHT = 640; // 实际在界面量出来的大概是600

static double KB = 1000;
static double MB = 1000 * KB;
static double GB = 1000 * MB;

static double KiB = 1024;
static double MiB = 1024 * KiB;
static double GiB = 1024 * MiB;

const QString BackupModuleName = "BackupModule";
const QString RestoreModuleName = "RestoreModule";
const QString AdvancedModuleName = "AdvancedModule";

static const QString UOS_RECOVERY_INI = "etc/uos-recovery/uos-recovery.ini";
static const QString BACKUP_GROUP = "backup";
static const QString RESTORE_GROUP = "restore";
static const QString GHOST_GROUP = "ghost";
static const QString GHOST_POLICY = "ghost_policy";
static const QString BACKUP_DEVICE_UUID_KEY = "device_uuid";
static const QString BACKUP_HISTORY_DEVICE_UUID_KEY = "history_device_uuid";
static const QString BACKUP_SYNC_TYPE_KEY = "type";
static const QString BACKUP_USER_DATA_SYNC_TYPE_KEY = "user_data_type";

static const QString RESTORE_AUTO_RESTORE_KEY = "auto_restore";
static const QString RESTORE_DO_RESTORE_KEY = "do_restore";
static const QString RESTORE_RECOVERY_TYPE_KEY = "recovery_type";
static const QString RESTORE_BACKUP_DEV_UUID_KEY = "backup_device_uuid";
static const QString RESTORE_BACKUP_POINT_KEY = "backup_point";
static const QString V20RESTORE_PATH = "v20_restore_path";
static const QString V20RESTORE_LIST = "v20_restore_list";
static const QString V20RESTORE_TAR = "v20_restore_tar";

static const QString LAYOUT_GROUP = "layout";
static const QString UI_LAYOUT_BACKUP = "ui_layout_backup";
static const QString UI_LAYOUT_RESTORE = "ui_layout_restore";
static const QString UI_LAYOUT_ADVANCE = "ui_layout_advance";

enum ErrorCode {
    InvalidClient = -2,
    UnKnow = -1,                       //未知错误
    OK = 0,
    PartitionNotSupportRecovery, //分区方式不支持备份还原
    InsufficientDiskSpace,       //磁盘空间不足
    InvalidDirectory,            //无效的目录
    NoAuthority,                 //没有权限
    NoSpecifySystem,              //未指定需要备份的系统
    NullPointer,
    PartitionNotMount,           //分区未挂载
    PartitionNotExist,            //分区不存在
    ParamError,                    //参数错误
    RootUuidIsEmpty,               //根分区uuid为空
    DestUuidIsEmpty,               // 存储目的分区uuid为空
    DbusError,
    InvalidVersion   // deepin-boot-kit get version failed
};

enum OperateStatus {
    Running = 0,
    Success,
    Failed,
    Abort
};

enum FileType {
    InvalidFileType = -1,
    Dir = 0,
    File
};

class FileItem;
struct FileItem {
    QString fileName;
    FileType fileType;
    quint64 size = 0;
    QList<FileItem> children;

    FileItem() : fileName(""), fileType(FileType::InvalidFileType)
    {
        children.clear();
    }

    QJsonObject marshal() {
        QJsonObject jsonObject;
        jsonObject.insert("fileName", fileName);
        jsonObject.insert("fileType", fileType);
        QJsonArray array;
        for (auto &item : children) {
            QJsonObject childObject;
            childObject.insert("fileName", fileName);
            childObject.insert("fileType", fileType);
            array.append(childObject);
        }
        jsonObject.insert("children", array);
        return jsonObject;
    }

    void unmarshal(QJsonObject &jsonObject) {
        fileName = jsonObject.value("fileName").toString();
        fileType = static_cast<FileType>(jsonObject.value("fileType").toInt());
        QJsonArray array = jsonObject.value("children").toArray();
        for (auto a : array) {
            FileItem childItem;
            childItem.fileName = a.toObject().value("fileName").toString();
            childItem.fileType = static_cast<FileType>(a.toObject().value("fileType").toInt());
            children.append(childItem);
        }
    }

    bool operator==(const FileItem &other)
    {
        return ((this->fileName == other.fileName) && (this->fileType == other.fileType));
    }
};

//备份还原策略类型
enum RecoveryType {
    InvalidRecoveryType = -1,
    Rsync = 0,                   //使用rsync进行数据备份和恢复
    OSTree,                      //使用OSTree进行系统版本管理
    BtrfsSnapshot,               //使用btrfs文件系统快照进行数据备份和恢复
    LvmSnapshot,
    RsyncV20
};

//备份还原操作类型
enum OperateType {
    Invalid = -1,
    SystemBackup = 0,            //系统备份
    SystemRestore,               //系统恢复
    UserDataBackup,              //用户数据备份
    UserDataRestore,             //用户数据恢复
    removeBackup,                //删除备份数据
    CheckFullSystemBackupSpace,  // 全量系统备份空间校验
    CheckIncSystemBackupSpace,   // 增量系统备份空间校验
    CheckGhostBackupSpace,       // Ghost备份空间校验
    CheckUserDataBackupSpace,    //数据备份空间校验
    CheckDimFileUseSpace,        //v20备份数据空间校验
    GhostBackup,                 //Ghost备份
    DimFileRestore,              //Dim文件转换
    AutoBackup,                  // OStree SubmissionType 为0时
    FactoryRestore               // 恢复出厂设置
};

// OStree 原子更新 commit 类型
enum CommitType {
    InvalidCommit = -1,
    SystemCommit = 0,      // 系统提交
    UserCommit = 1,        // 用户提交
    InstallerCommit = 2    // 安装器提交
};

// 架构类型
enum PlatformType {
    X86_64 = 0,
    I386,
    I686,
    AMD64,
    X86,
    SW_64,
    MIPS64,
    LOONGARCH64,
    AARCH64,
    UNKNOW
};

//用户操作记录
struct OperateRecord {
    QString operateID;           //操作id（UUID）
    QString username;            //操作者用户名
    quint64 startTime;           //操作开始时间
    quint64 endTime;             //操作结束时间
    OperateType operateType;     //操作类型
    RecoveryType recoveryType;   //策略类型
};
typedef QList<OperateRecord> OperateRecordList;

//分区信息
struct Partition {
    QString name;
    QString fsType;
    QString label;
    QString uuid;
    QString deviceType;
    quint64 size = 0;
    quint64 used = 0;
    quint64 free = 0;
    bool fsTypeSupported; // 是否支持系统备份
    bool fsTypeSupportedDataBackup; // 是否支持数据备份
    bool externalDevice;

    QJsonObject marshal() {
        QJsonObject jsonObject;
        jsonObject.insert("name", name);
        jsonObject.insert("fsType", fsType);
        jsonObject.insert("label", label);
        jsonObject.insert("uuid", uuid);
        jsonObject.insert("deviceType", deviceType);
        jsonObject.insert("size", QString::number(size));
        jsonObject.insert("used", QString::number(used));
        jsonObject.insert("free", QString::number(free));
        jsonObject.insert("fsTypeSupported", fsTypeSupported);
        jsonObject.insert("fsTypeSupportedDataBackup", fsTypeSupportedDataBackup);
        jsonObject.insert("externalDevice", externalDevice);
        return jsonObject;
    }

    void unmarshal(const QJsonObject &jsonObject) {
        name = jsonObject.value("name").toString();
        fsType = jsonObject.value("fsType").toString();
        label = jsonObject.value("label").toString();
        uuid = jsonObject.value("uuid").toString();
        deviceType = jsonObject.value("deviceType").toString();
        size = jsonObject.value("size").toString().toULongLong();
        used = jsonObject.value("used").toString().toULongLong();
        free = jsonObject.value("free").toString().toULongLong();
        fsTypeSupported = jsonObject.value("fsTypeSupported").toBool();
        fsTypeSupportedDataBackup = jsonObject.value("fsTypeSupportedDataBackup").toBool();
        externalDevice = jsonObject.value("externalDevice").toBool();
    }

};

//备份信息
struct BackupInfo {
    QString operateID;           //操作id（UUID）
    QString username;            //操作者用户名
    quint64 startTime = 0;       //操作开始时间
    QString remark;              //备注信息
    OperateType operateType = OperateType::Invalid;     //操作类型
    RecoveryType recoveryType = RecoveryType::InvalidRecoveryType;   //策略类型
    QString backupPath;          //备份路径
    quint64 size = 0;            //大小
    QString rootUUID;
    QString backupDevUUID;       //存储备份文件的分区
    QString backupDevice;
    bool backupDeviceRemovable = false;
    OperateStatus status = OperateStatus::Failed;        //状态
    quint64 submissionTime = 0;  // OStree: 十位时间戳，精度为秒
    QString systemVersion;       // OStree: 当前系统版本号
    QString submissionVersion;   // OStree: 提交版本号
    CommitType submissionType = CommitType::InvalidCommit;   // OStree: 本次提交类型
    QString backupVersion;
    QString versionDisplay;

    QJsonObject marshal() {
        QJsonObject jsonObject;
        jsonObject.insert("operateID", operateID);
        jsonObject.insert("username", username);
        jsonObject.insert("startTime", QString("%1").arg(startTime));
        jsonObject.insert("remark", remark);
        jsonObject.insert("operateType", operateType);
        jsonObject.insert("recoveryType", recoveryType);
        jsonObject.insert("backupPath", backupPath);
        jsonObject.insert("size", QString("%1").arg(size));
        jsonObject.insert("rootUUID", rootUUID);
        jsonObject.insert("backupDevUUID", backupDevUUID);
        jsonObject.insert("backupDevice", backupDevice);
        jsonObject.insert("backupDeviceRemovable", backupDeviceRemovable);
        jsonObject.insert("status", status);
        jsonObject.insert("SubmissionTime", QString("%1").arg(submissionTime));
        jsonObject.insert("SystemVersion", systemVersion);
        jsonObject.insert("SubmissionVersion", submissionVersion);
        jsonObject.insert("SubmissionType", submissionType);
        jsonObject.insert("backupVersion", backupVersion);
        jsonObject.insert("versionDisplay", versionDisplay);
        return jsonObject;
    }

    void unmarshal(const QJsonObject &jsonObject) {
        operateID = jsonObject.value("operateID").toString();
        username = jsonObject.value("username").toString();
        startTime = jsonObject.value("startTime").toString().toULongLong();
        remark = jsonObject.value("remark").toString();
        operateType = static_cast<OperateType>(jsonObject.value("operateType").toInt());
        recoveryType = static_cast<RecoveryType>(jsonObject.value("recoveryType").toInt());
        backupPath = jsonObject.value("backupPath").toString();
        size = jsonObject.value("size").toString().toULongLong();
        rootUUID = jsonObject.value("rootUUID").toString();
        backupDevUUID = jsonObject.value("backupDevUUID").toString();
        backupDevice = jsonObject.value("backupDevice").toString();
        backupDeviceRemovable = jsonObject.value("backupDeviceRemovable").toBool();
        status = static_cast<OperateStatus>(jsonObject.value("status").toInt());

        // OStree 才需要下面的字段
        if (jsonObject.contains("SubmissionTime")) {
            submissionTime = jsonObject.value("SubmissionTime").toString().toULongLong();
        }

        if (jsonObject.contains("SystemVersion")) {
            systemVersion = jsonObject.value("SystemVersion").toString();
        }

        if (jsonObject.contains("SubmissionVersion")) {
            submissionVersion = jsonObject.value("SubmissionVersion").toString();
        }

        if (jsonObject.contains("SubmissionType")) {
            submissionType = static_cast<CommitType>(jsonObject.value("SubmissionType").toInt(-1));
        }

        if (jsonObject.contains("backupVersion")) {
            backupVersion = jsonObject.value("backupVersion").toString();
        }

        if (jsonObject.contains("versionDisplay")) {
            versionDisplay = jsonObject.value("versionDisplay").toString();
        }
    }

    bool operator < (const BackupInfo& d) const {
        return startTime <= d.startTime;
    }
};
typedef QList<BackupInfo> BackupInfoList;

struct SystemBackupRequest {
    QString username;
    QString rootUUID;
    QString destUUID;
    QString remark;

    QJsonObject marshal() {
        QJsonObject jsonObject;
        jsonObject.insert("username", username);
        jsonObject.insert("rootUUID", rootUUID);
        jsonObject.insert("destUUID", destUUID);
        jsonObject.insert("remark", remark);
        return jsonObject;
    }

    void unmarshal(QJsonObject &jsonObject) {
        username = jsonObject.value("username").toString();
        rootUUID = jsonObject.value("rootUUID").toString();
        destUUID = jsonObject.value("destUUID").toString();
        remark = jsonObject.value("remark").toString();
    }
};

struct SystemRestoreRequest {
    QString username;
    BackupInfo backupInfo;

    QJsonObject marshal() {
        QJsonObject jsonObject;
        jsonObject.insert("username", username);
        QJsonObject backupObject = backupInfo.marshal();
        jsonObject.insert("backupInfo", backupObject);
        return jsonObject;
    }

    void unmarshal(QJsonObject &jsonObject) {
        username = jsonObject.value("username").toString();
        QJsonObject backupObject = jsonObject.value("backupInfo").toObject();
        backupInfo.unmarshal(backupObject);
    }
};

struct UserDataBackupRequest {
    QString username;
    QString rootUUID;
    QString destUUID;
    QString remark;
    QStringList exclude;

    QJsonObject marshal() {
        QJsonObject jsonObject;
        jsonObject.insert("username", username);
        jsonObject.insert("rootUUID", rootUUID);
        jsonObject.insert("destUUID", destUUID);
        jsonObject.insert("remark", remark);
        QJsonArray array;
        for (auto &i : exclude) {
            array.append(i);
        }
        jsonObject.insert("exclude", array);
        return jsonObject;
    }

    void unmarshal(QJsonObject &jsonObject) {
        username = jsonObject.value("username").toString();
        rootUUID = jsonObject.value("rootUUID").toString();
        destUUID = jsonObject.value("destUUID").toString();
        remark = jsonObject.value("remark").toString();
        QJsonArray array = jsonObject.value("exclude").toArray();
        for (auto a : array) {
            exclude.append(a.toString());
        }
    }
};

struct UserDataRestoreRequest {
    QString username;
    BackupInfo backupInfo;
    QJsonObject marshal() {
        QJsonObject jsonObject;
        jsonObject.insert("username", username);
        QJsonObject backupObject = backupInfo.marshal();
        jsonObject.insert("backupInfo", backupObject);
        return jsonObject;
    }

    void unmarshal(QJsonObject &jsonObject) {
        username = jsonObject.value("username").toString();
        QJsonObject backupObject = jsonObject.value("backupInfo").toObject();
        backupInfo.unmarshal(backupObject);
    }
};

typedef UserDataRestoreRequest RemoveUserDataBackupRequest;

static const int LOG_MASK_BEGIN_TIME = 0x01;
static const int LOG_MASK_END_TIME = 0x02;
static const int LOG_MASK_OPERATE_TYPE = 0x04;
static const int LOG_MASK_USER_NAME = 0x08;
static const int LOG_MASK_REMARK = 0x10;

struct OperateLogQuery {
    int mask = -1;
    quint64 beginTime = 0;
    quint64 endTime = 0;
    OperateType operateType;
    QString username;
    QString remark;
    bool orderByTimeDesc = true;

    QJsonObject marshal() {
        QJsonObject jsonObject;
        jsonObject.insert("mask", mask);
        jsonObject.insert("beginTime", QString("%1").arg(beginTime));
        jsonObject.insert("endTime", QString("%1").arg(endTime));
        jsonObject.insert("operateType", operateType);
        jsonObject.insert("username", username);
        jsonObject.insert("remark", remark);
        jsonObject.insert("orderByTimeDesc", orderByTimeDesc);
        return jsonObject;
    }

    void unmarshal(QJsonObject &jsonObject) {
        mask = jsonObject.value("mask").toInt();
        beginTime = jsonObject.value("beginTime").toString().toULongLong();
        endTime = jsonObject.value("endTime").toString().toULongLong();
        operateType = static_cast<OperateType>(jsonObject.value("operateType").toInt());
        username = jsonObject.value("username").toString();
        remark = jsonObject.value("remark").toString();
        orderByTimeDesc = jsonObject.value("orderByTimeDesc").toBool();
    }
};

struct OperateLog {
    quint64 time = 0;
    OperateType operateType;
    QString username;
    OperateStatus status;
    QString remark;
    QString reserve;

    QJsonObject marshal() {
        QJsonObject jsonObject;
        jsonObject.insert("time", QString("%1").arg(time));
        jsonObject.insert("operateType", operateType);
        jsonObject.insert("username", username);
        jsonObject.insert("status", status);
        jsonObject.insert("remark", remark);
        jsonObject.insert("reserve", reserve);
        return jsonObject;
    }

    void unmarshal(const QJsonObject &jsonObject) {
        time = jsonObject.value("time").toString().toULongLong();
        operateType = static_cast<OperateType>(jsonObject.value("operateType").toInt());
        username = jsonObject.value("username").toString();
        status = static_cast<OperateStatus>(jsonObject.value("status").toInt());
        remark = jsonObject.value("remark").toString();
        reserve = jsonObject.value("reserve").toString();
    }
};

typedef QList<OperateLog> OperateLogList;

static QString getRecoveryType(RecoveryType recoveryType)
{
    static QMap<RecoveryType, QString> all = {
            {RecoveryType::Rsync, "rsync"},
            {RecoveryType::OSTree, "ostree"},
            {RecoveryType::BtrfsSnapshot, "btrfsSnapshot"},
            {RecoveryType::LvmSnapshot, "LvmSnapshot"},
            {RecoveryType::RsyncV20, "RsyncV20"}
    };

    if (all.contains(recoveryType)) {
        return all.value(recoveryType);
    }

    return QString("");
}
