#include "UImgOperate.h"
#include "Process.h"
#include "SquashfsTask.h"
#include "RsyncTask.h"
#include "FsTab.h"
#include "utils/Utils.h"
#include "utils/global.h"
#include <QDir>
#include <QFile>
#include <QJsonDocument>
#include <QJsonObject>
#include <QJsonArray>
#include <QJsonValue>
#include <QSettings>
#include <QCoreApplication>
#include <QDebug>

static const QString SystemUpgradeMountPoint = "/SystemUpgradeMountPoint";
static const QString SquashFileMountDir = "/mnt";
static const QString TargetName = "/target";
static const QString TmpFileDir = "/tmp";
static const QString InfoFileName = "info.json";
static const QString PolicyFileName = "partition_policy.json";
static const qint64 RangeSpace = GiB;
static const QString AnotherRootMountPoint = "/sysRootb";
static const int OsMajorVersionV20 = 20;

UImgOperate::UImgOperate()
{
    m_rsyncTask = new RsyncTask();
    connect(m_rsyncTask, &RsyncTask::progressChanged, this, &UImgOperate::progressChangeSlot);
    connect(m_rsyncTask, &RsyncTask::error, this, &UImgOperate::doneError);
    connect(m_rsyncTask, &RsyncTask::success, this, &UImgOperate::doneSuccess);
}

UImgOperate::~UImgOperate()
{
    if (m_rsyncTask != nullptr) {
        delete m_rsyncTask;
        m_rsyncTask = nullptr;
    }
}

UImgOperate::CheckFileExitCode UImgOperate::checkImgFile(const QString& imgFile, const QStringList &destDevices)
{
    m_ghostDiskInfoList.clear();
    GhostCheckItem src;
    GhostCheckItem dest;
    QJsonDocument partitionDoc;
    if (!Utils::readJsonFile(SquashFileMountDir + "/" + PolicyFileName, partitionDoc)) {
        qWarning() << "checkImgFile read " << SquashFileMountDir + "/" + PolicyFileName << " failed !";
        return CanNotInstall;
    }

    QSet<QString> deviceSet;
    QJsonArray partitionArray = partitionDoc.array();
    int partitionSize = partitionArray.size();
    for (int i = 0; i < partitionSize; ++i) {
        QJsonObject partitionObjItem = partitionArray.at(i).toObject();
        if (partitionObjItem.contains("device")) {
            QString device = partitionObjItem.value("device").toString();
            if (!device.isEmpty()) {
                deviceSet.insert(device);
            }
        }
    }
    int ghostDeviceNum = deviceSet.size();

    if (destDevices.size() != ghostDeviceNum) {
        qWarning() << "checkImgFile ghostDeviceNum = "<<ghostDeviceNum<< ", destDevices.size() = "<<destDevices.size();
        return CanNotInstall;
    }

    QJsonDocument infoDoc;
    if (!Utils::readJsonFile(SquashFileMountDir + "/" + InfoFileName, infoDoc)) {
        qWarning() << "checkImgFile read " << SquashFileMountDir + "/" + InfoFileName << " failed !";
        return CanNotInstall;
    }

    SquashfsTask squashfsTask;
    QJsonObject infosObjFromFile = infoDoc.object();
    //QJsonObject infosObj = squashfsTask.getDeviceInfos();

    QString err = "";
    // 检查所选的磁盘中是否包含镜像文件，如果包含则返回CanNotInstall
    if (!checkImgFileInDisk(imgFile, destDevices, err)) {
        qWarning() << "checkImgFileInDisk failed ! err is : " << err;
        return CanNotInstall;
    }

    // 检查磁盘大小
    quint64 diskTotalSize = infosObjFromFile.value("diskSize").toString().toULongLong();
    if (!checkDiskSize(diskTotalSize, destDevices, err)) {
        qWarning() << "checkDiskSize failed ! err is : " << err<<", diskTotalSize = "<<diskTotalSize;
        return CanNotInstall;
    }

    if (infosObjFromFile.contains("diskInfo")) {
        QList<GhostDiskInfo> srcDiskInfoList;
        QJsonArray sysDiskJsonArray = infosObjFromFile.value("diskInfo").toArray();
        int sysDiskNum = sysDiskJsonArray.size();
        qInfo()<<"check each disk size by diskInfo, sysDiskNum = "<<sysDiskNum;
        for (int i = 0; i < sysDiskNum; ++i) {
            QJsonObject sysDisk = sysDiskJsonArray.at(i).toObject();
            GhostDiskInfo gDisk;
            gDisk.unmarshal(sysDisk);
            srcDiskInfoList.append(gDisk);
        }
        m_ghostDiskInfoList = srcDiskInfoList;

        if (!checkDiskSize(srcDiskInfoList, destDevices, src, dest)) {
            qWarning() << "checkDiskSize failed, src.diskSize = "<<src.diskSize<<", dest.diskSize = "<<dest.diskSize;
            return CanNotInstall;
        }
    }

    // 检查引导系统类型是否匹配
    if (infosObjFromFile.value("isEFI").toBool() != Utils::isEFIMode()) {
        qWarning() << "checkBootStrapType failed !";
        return CanNotInstall;
    }

    // 检查img文件（架构、CPU型号、显卡型号）
    QStringList infoList = Utils::getCmdListReturn("uname -m");
    if (infoList.size() > 0) {
        if (infosObjFromFile.value("platform").toString().compare(infoList.first())) {
            qWarning() << "platform not match !";
            return CanNotInstall;
        }
    }

    QString vga;
    Utils::getVGAFromLshw(vga);
    if (infosObjFromFile.value("displayDriver").toString().compare(vga)) {
        qWarning() << "displayDriver not match !, cur vga = "<<vga;
        return CarefullyInstall;
    }

    CpuInfo cpu;
    Utils::getCpuInfo(cpu);
    QString ghostCpuModel = infosObjFromFile.value("cpuModel").toString();
    if (infosObjFromFile.value("cpuModel").toString().compare(cpu.modelName)) {
        qWarning() << "cpuModel not match !, cur cpu mode name = "<<cpu.modelName<<", ghostCpuModel = "<<ghostCpuModel;
        return CarefullyInstall;
    }

    return CanInstall;
}

UImgOperate::InstallFileExitCode UImgOperate::installImgFile(const QString& imgFile, const QStringList &destDevices)
{
    CheckFileExitCode checkCode = checkImgFile(imgFile, destDevices);
    if (CanNotInstall == checkCode) {
        qWarning() << "do checkImgFile before install failed ! err : " << checkCode;
        return UnknowError;
    }

    QString cmdLog = "";

    // 拷贝info.json文件到/tpm/info.json
    QFile::remove(TmpFileDir + "/" + InfoFileName);
    if (!QFile::copy(SquashFileMountDir + "/" + InfoFileName, TmpFileDir + "/" + InfoFileName)) {
        qWarning() << "copy info.json to /tpm/info.json failed !";
        return CopyConfigFailed;
    }

    int osMajorVer = getOSMajorVersion(SquashFileMountDir + "/localhost");
    QString destDeepinInstallerConf = "/etc/deepin-installer/deepin-installer.conf";
    QString srcDeepinInstallerConf = SquashFileMountDir + "/localhost" + destDeepinInstallerConf;
    if (OsMajorVersionV20 == osMajorVer) {
        QString v20DeepinInstallerConf = "/etc/deepin-installer.conf";
        srcDeepinInstallerConf = SquashFileMountDir + "/localhost" + v20DeepinInstallerConf;
    }
    qInfo()<<"installImgFile, osMajorVer = "<<osMajorVer<<", srcDeepinInstallerConf = "<<srcDeepinInstallerConf;
    QFile::remove(destDeepinInstallerConf);
    if (!QFile::copy(srcDeepinInstallerConf, destDeepinInstallerConf)) {
        qWarning() << "failed to copy src = "<<srcDeepinInstallerConf<<", to dest = "<<destDeepinInstallerConf;
        return CopyConfigFailed;
    }

    // 拷贝partition_policy.json文件到/tpm/partition_policy.json
    QFile::remove(TmpFileDir + "/" + PolicyFileName);
    if (!QFile::copy(SquashFileMountDir + "/" + PolicyFileName, TmpFileDir + "/" + PolicyFileName)) {
        qWarning() << "copy partition_policy.json to /tpm/partition_policy.json failed !";
        return CopyConfigFailed;
    }

    // 根据本机磁盘设备更新partition_policy.json文件中的device字段，对于要使用多盘的情况，也要更新分区的起止位置信息
    if (!updatePolicyDeviceInfo(cmdLog)) {
        qWarning() << cmdLog;
        return UnknowError;
    }

    QMap<QString, QString> pvVgMaps;
    if (Device::getPvDisplay(pvVgMaps)) {
    //    Device::deleteLVM(destDevices, pvVgMaps);
        Device::deleteVG(destDevices, pvVgMaps);
    }

    QStringList sysVgNameList;
    if (Device::getVGNames(sysVgNameList)) {
        QString partitionFile = TmpFileDir + "/" + PolicyFileName;
        if (!this->renameVgName(partitionFile, sysVgNameList)) {
            qCritical()<<"renameVgName failed, partitionFile = "<<partitionFile<<", sysVgNameList = "<<sysVgNameList;
            return UnknowError;
        }
    }

    // 使用deepin-installer-parted处理分区操作
    if (!doPart(cmdLog)) {
        qWarning() << cmdLog;
        return DoPartFailed;
    }

    // 执行bind操作
    if (!doBind(cmdLog)) {
        qWarning() << cmdLog;
        return DoBindFailed;
    }

    // 设置同步文件系统
    m_rsyncTask->setOptions({"-avAXHSi", "--stats", "--info=progress2", "--ignore-missing-args"});
    m_rsyncTask->setSourcePath(SquashFileMountDir + "/localhost/");
    m_rsyncTask->setDestinationPath("/target");
    m_rsyncTask->setExcludes({});
    m_rsyncTask->enableDryRun(false);
    m_rsyncTask->buildArgumentsForBackup();
    m_rsyncTask->start();

    return Installing;
}

bool UImgOperate::mountFile(const QString &imgFile)
{
    QString cmd = QString("mount %1 %2").arg(imgFile).arg(SquashFileMountDir);
    QString cmdLog = "";
    QString err = "";
    if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, err)) {
        qInfo()<<"mountFile failed, cmd = "<<cmd<<", err = "<<err;
        return false;
    }

    return true;
}

bool UImgOperate::umountFile()
{
    QString cmd = QString("umount %1").arg(SquashFileMountDir);
    QString cmdLog = "";
    QString err = "";
    if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, err)) {
        qInfo()<<"umountFile failed, cmd = "<<cmd<<", err = "<<err;
        return false;
    }

    return true;
}

void UImgOperate::progressChangeSlot(const QJsonObject &value)
{
    qInfo() << "process value is : " << value;
}

void UImgOperate::doneError()
{
    qWarning() << "process failed";
    qApp->exit(CopyFailed);
}

void UImgOperate::doneSuccess()
{
    qInfo() << "process success";
    QString err = "";
    if (!updateFstab(err)) {
        qWarning() << err;
        qApp->exit(UpdateFstabFailed);
    }

    if (!UImgOperate::updateCrypttab()) {
        qApp->exit(UpdateCrypttabFailed);
    }

    if (!updateGrub(err)) {
        qWarning() << err;
        qApp->exit(UpdateGrubFailed);
    }

    if (!backUpConfFiles(err)) {
        qWarning() << err;
        qApp->exit(SetupUDisksRulesFailed);
    }

    qApp->exit(InstallSuccess);
}

bool UImgOperate::updateCrypttab()
{
    /*
     * zxm@zxm-PC:~$ lsblk -lfo PATH,FSTYPE | grep -i "crypto_LUKS" | awk '{print $1}'
          /dev/sda3
          /dev/sdb1
     */
    QString cmd = QString("lsblk -lfo PATH,FSTYPE | grep -i \"crypto_LUKS\" | awk '{print $1}'");
    QString output = "";
    QString err;
    if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, output, err)) {
        qInfo()<<"updateCrypttab cryptPartition failed, err = "<<err;
        return false;
    }

    if (output.isEmpty() && err.isEmpty()) {
        return true; // not encrypt, do nothing
    }

    QMap<QString, QString> cryptUuidNameMap;
#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
    QStringList cryptPartitionList = output.split("\n", Qt::SkipEmptyParts);
#else
    QStringList cryptPartitionList = output.split("\n", QString::SkipEmptyParts);
#endif
    for (auto &partition : cryptPartitionList) {
        partition = partition.trimmed();
        output.clear();
        /*
         * zxm@zxm-PC:~$ lsblk /dev/sda3 -lfo UUID,FSTYPE | grep -i "crypto_LUKS" | awk '{print $1}'
              48a1b5e5-631f-40f3-9bc7-d8921515535a
         */
        cmd = QString("lsblk %1 -lfo UUID,FSTYPE | grep -i \"crypto_LUKS\" | awk '{print $1}'").arg(partition);
        if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, output, err)) {
            qInfo()<<"updateCrypttab crypto_LUKS uuid failed, err = "<<err;
            return false;
        }

        QString cryptoLuksUUID = output.trimmed();

        /*
         * zxm@zxm-PC:~$ lsblk /dev/sda3 -lfo NAME | grep -i "luks_crypt" | awk '{print $1}'
             luks_crypt0
         */
        output.clear();
        cmd = QString("lsblk %1 -lfo NAME | grep -i \"luks_crypt\" | awk '{print $1}'").arg(partition);
        if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, output, err)) {
            qInfo()<<"updateCrypttab crypto_LUKS name failed, err = "<<err;
            return false;
        }

        QString cryptoLuksName = output.trimmed();
        cryptUuidNameMap.insert(cryptoLuksUUID, cryptoLuksName);
    }

    QString crypttabFilePath = TargetName + "/etc/crypttab";
// #ifdef QT_DEBUG
//     crypttabFilePath = "/home/zxm/Desktop/crypttab";
// #endif
    QFile rCrypttabFile(crypttabFilePath);
    if (!rCrypttabFile.open(QIODevice::Text | QIODevice::ReadOnly)) {
        qCritical()<<"updateCrypttab error : failed to read file "<<crypttabFilePath;
        return false;
    }
    QString data = rCrypttabFile.readAll();
    rCrypttabFile.close();
#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
    QStringList dataList = data.split("\n", Qt::SkipEmptyParts);
#else
    QStringList dataList = data.split("\n", QString::SkipEmptyParts);
#endif
    QString comments;
    QString options;
    for (auto &cryptData : dataList) {
        if (cryptData.startsWith("#")) {
            comments = cryptData + "\n";
            continue;
        }

        if (options.isEmpty() && cryptData.startsWith("luks_crypt")) {
            options = cryptData.right(cryptData.length() - cryptData.indexOf("UUID="));
            options = options.right(options.length() - options.indexOf(" ")) + "\n";
        }
    }

    QFile wCrypttabFile(crypttabFilePath);
    if (!wCrypttabFile.open(QIODevice::Truncate | QIODevice::Text | QIODevice::WriteOnly)) {
        qCritical()<<"updateCrypttab error : failed to open write file "<<crypttabFilePath;
        return false;
    }
    if (!comments.isEmpty()) {
        wCrypttabFile.write(comments.toStdString().c_str());
    }
    for (auto &uuid : cryptUuidNameMap.keys()) {
        QString line = cryptUuidNameMap[uuid] + "  UUID=" + uuid + " " + options;
        wCrypttabFile.write(line.toStdString().c_str());
    }

    wCrypttabFile.close();
    return true;
}

QMap<QString, DeviceInfoPtr> UImgOperate::matchDevices(QList<GhostDiskInfo> &gDiskList, DeviceInfoList &installDeviceList)
{
    QMap<QString, DeviceInfoPtr> ghostDevName2InstallDeviceMap;
    int diskNum = gDiskList.size();
    if (diskNum != installDeviceList.size()) {
        qWarning()<<"matchDevices: disk size not same";
        return ghostDevName2InstallDeviceMap;
    }

    std::sort(gDiskList.begin(), gDiskList.end());
    std::sort(installDeviceList.begin(), installDeviceList.end(), deviceInfoTotalSizeCompare);
    for (int i = 0; i < diskNum; ++i) {
        auto &gDisk = gDiskList.at(i);
        auto &installDev = installDeviceList.at(i);
        if (gDisk.totalSizeBytes <= installDev->sizeBytes) {
            ghostDevName2InstallDeviceMap.insert(gDisk.deviceName, installDev);
        }
    }

    return ghostDevName2InstallDeviceMap;
}

bool UImgOperate::updatePolicyDeviceInfo(QString &err)
{
    QString policyFilePath = TmpFileDir + "/" + PolicyFileName;

    QJsonDocument policyDoc;
    if (!Utils::readJsonFile(policyFilePath, policyDoc)) {
        err = "updatePolicyDeviceInfo read " + policyFilePath + " failed !";
        return false;
    }

    QJsonArray policyArray = policyDoc.array();
    QJsonArray policyArrayOut;

    // 添加需要删除的操作列表
    QJsonArray deletePartList;
    if (!getDeletePartList(deletePartList, err)) {
        err = "updatePolicyDeviceInfo getDeletePartList failed ! err : " + err;
        return false;
    }
    for (int i = 0; i < deletePartList.size(); i++ ) {
        policyArrayOut.append(deletePartList.at(i).toObject());
    }

    // 添加需要编辑的操作列表
    // 根据引导模式确定即将使用的磁盘格式
    QString diskTypeStr = "msdos";
    QString isSWStr = "";
    err = "";
    if (!Process::spawnCmd("/bin/bash", {"-c", "uname -m"}, isSWStr, err)) {
        err = "updatePolicyDeviceInfo uname -m failed ! err : " + err;
        return false;
    }
    if (QFile::exists("/sys/firmware/efi") || isSWStr.contains("sw")) {
        diskTypeStr = "gpt";
    }
    for (QString devPath : m_destDevices) {
        QJsonObject policyObjOutItem;
        policyObjOutItem.insert("device", devPath);
        policyObjOutItem.insert("diskType", diskTypeStr);
        policyObjOutItem.insert("operate", "edit");
        policyObjOutItem.insert("type", "disk");
        policyArrayOut.append(policyObjOutItem);
    }

    Device devToUse;
    DeviceInfoList devicesToUse = devToUse.getDeviceByLsblk();
    DeviceInfoList devicesToUseList;
    for (DeviceInfoPtr deviceItem : devicesToUse) {
        if (m_destDevices.contains(deviceItem->path)) {
            devicesToUseList.append(deviceItem);
        }
    }

    int devSize = devicesToUseList.size();
    int ghostDiskSize = m_ghostDiskInfoList.size();
    if (devSize != m_ghostDiskInfoList.size()) {
        qCritical()<<"updatePolicyDeviceInfo error: devSize = "<<devSize<<", ghostDiskSize = "<<ghostDiskSize;
        return false;
    }
    auto devName2InstallDeviceMap = UImgOperate::matchDevices(m_ghostDiskInfoList, devicesToUseList);
    for (int i = 0; i < devSize; ++i) {
        QString devPath = devicesToUseList.at(i)->path;
        qInfo()<<"updatePolicyDeviceInfo, i = "<<i<<", devPath = "<<devPath<<", devSize = "<<devSize;
    }

    int avalibleRealDeviceSize = devicesToUseList.size();
    QJsonObject firstDiskObj = policyArray.first().toObject();
    QString preDevice = firstDiskObj.value("device").toString();
    bool isSameDisk = true;

    qint64 totalPartSize = 0;
    int devicesToUseIndex = 0;
    for (int i = 0; i < policyArray.size(); i++) {
        QJsonObject policyObjItem = policyArray.at(i).toObject();
        QString curDevice = policyObjItem.value("device").toString();
        if (curDevice != preDevice) {
            isSameDisk = false;
            ++devicesToUseIndex;
            qInfo() << "devicesToUseIndex = " << devicesToUseIndex << ", curDevice = " << curDevice
                    << ", preDevice = " << preDevice;
            preDevice = curDevice;
            if (devicesToUseIndex > avalibleRealDeviceSize) {
                err = "updatePolicyDeviceInfo devicesToUseIndex is out of avalibleRealDeviceSize !";
                return false;
            }
        }

        QJsonObject policyArrayOutObj = policyArrayOut.at(devicesToUseIndex).toObject();

        QString partitionType = policyObjItem.value("partType").toString();

        if (policyObjItem.value("filesystem").toString().contains("swap")) {
            policyObjItem["filesystem"] = "linux-swap";
            policyObjItem["mountPoint"] = "swap";
        }
        if ((!policyObjItem.value("type").toString().compare("disk"))
            || (!policyObjItem.value("type").toString().compare("partition"))) {
            // 检测分区的终点与设备终点位置的距离，小于设置的阈值，则调整分区大小使用设备的全部空间
            DeviceInfoPtr installDevice = devName2InstallDeviceMap[curDevice];
            if (nullptr == installDevice) {
                qInfo()<<"updatePolicyDeviceInfo error, get nullptr by curDevice = "<<curDevice;
                return false;
            }

            if (partitionType != "extended") {
                totalPartSize += policyObjItem.value("size").toVariant().toLongLong() * MiB;
                if (!updatePartStartEndPoint(policyObjItem, policyArrayOutObj, totalPartSize,
                                             installDevice->sizeBytes, installDevice->phySectorSize, isSameDisk)) {
                    totalPartSize = 0;
                    devicesToUseIndex++;
                    if (devicesToUseIndex >= devicesToUseList.size()) {
                        err = "updatePolicyDeviceInfo devicesToUseIndex is out of devicesToUseList !";
                        QString policyItem = Utils::JsonToQString(policyObjItem);
                        qInfo()<<"updatePolicyDeviceInfo: devicesToUseIndex = "<<devicesToUseIndex
                            <<", devicesToUseList.size() = "<<devicesToUseList.size()<<", partitionType = "
                            <<partitionType<<", policyItem = "<<policyItem;
                        return false;
                    }
                }
            }

            // 为分区item指定设备
            policyObjItem["device"] = installDevice->path;
            QString policyObjItemStr = Utils::JsonToQString(policyObjItem);
            qInfo()<<"updatePolicyDeviceInfo, policyObjItemStr = "<<policyObjItemStr;
        }
        policyArrayOut.append(policyObjItem);
    }

    QJsonDocument policyDocOut(policyArrayOut);
    QFile file(policyFilePath);
    if (!file.open(QIODevice::Text | QIODevice::WriteOnly)) {
        err = "ghost install updatePolicyDeviceInfo error : open file " + policyFilePath + " failed";
        return false;
    }
    file.write(policyDocOut.toJson());
    file.close();

    return true;
}

bool UImgOperate::doPart(QString &err)
{
    // 设置deepin-installer-parted要使用的环境变量
    QString diConfigFileEnv = "/etc/deepin-installer/deepin-installer.conf";

    // 调用deepin-installer-parted执行分区操作，程序中已经包含了磁盘格式化操作
    QString partedCmd = QString("deepin-installer-parted -m 0 -c %1").arg(TmpFileDir + "/" + PolicyFileName);
    QString cmdLog = "";
    if (!Process::spawnCmd("/bin/bash", {"-c", partedCmd}, cmdLog, err)) {
        err = "ghost install parted error cmd : " + partedCmd + " err : " + err;
        return false;
    }

    // 执行分区挂载操作
    QSettings settings(diConfigFileEnv, QSettings::IniFormat);
    QString mountPoints = settings.value("DI_MOUNTS_POINTS").toString();
    QString mountPartCmd = QString("deepin-installer-parted -m 4 -c \"%1\"").arg(mountPoints);
    cmdLog = "";
    err = "";
    if (!Process::spawnCmd("/bin/bash", {"-c", mountPartCmd}, cmdLog, err)) {
        err = "ghost install parted error cmd : " + mountPartCmd + " err : " + err;
        return false;
    }

//    // TODO: 处理EFI分区使能和swap分区激活
//    QString espOnCmd = QString("parted -s %1 set 1 esp on").arg(m_destDevices.first());
//    cmdLog = "";
//    err = "";
//    if (!Process::spawnCmd("/bin/bash", {"-c", espOnCmd}, cmdLog, err)) {
//        err = "ghost install esp on error cmd : " + espOnCmd + " err : " + err;
//        return false;
//    }

    return true;
}

bool UImgOperate::doBind(QString &err)
{
    QJsonDocument infoDoc;
    if (!Utils::readJsonFile(TmpFileDir + "/" + InfoFileName, infoDoc)) {
        err = "doBind read " + TmpFileDir + "/" + InfoFileName + " failed !";
        return false;
    }

    // 根据fstab信息绑定目录
    QJsonArray bindInfoArray = infoDoc.object().value("fstab").toArray();
    QDir mkDirs;
    for (int i = 0; i < bindInfoArray.size(); i++) {
        QJsonObject bindInfoObjItem = bindInfoArray.at(i).toObject();
        if (bindInfoObjItem.value("options").toString().contains("bind")) {
            QString srcBindDir = bindInfoObjItem.value("device").toString();
            QString destBindDir = bindInfoObjItem.value("mountPoint").toString();
            mkDirs.mkpath(TargetName + srcBindDir);
            mkDirs.mkpath(TargetName + destBindDir);
            QString cmd = QString("mount -v --bind %1%2 %3%4").arg(TargetName).arg(srcBindDir).arg(TargetName).arg(destBindDir);
            QString cmdLog = "";
            err = "";
            if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, err)) {
                err = "ghost install bind error cmd : " + cmd + " err : " + err;
                return false;
            }
        }
    }

    // 绑定固定目录
    QMap<QString, QString> bindDirs = {
            {TargetName + "/dev", QString("mount -v --bind /dev/ %1/dev").arg(TargetName)},
            {TargetName + "/dev/pts", QString("mount -vt devpts devpts %1/dev/pts").arg(TargetName)},
            {TargetName + "/proc", QString("mount -vt proc proc %1/proc").arg(TargetName)},
            {TargetName + "/sys", QString("mount -vt sysfs sysfs %1/sys").arg(TargetName)},
            {TargetName + "/run", QString("mount --bind /run %1/run").arg(TargetName)}
    };

    for (QString dirItem : bindDirs.keys()) {
        mkDirs.mkpath(dirItem);
        QString cmdLog = "";
        err = "";
        if (!Process::spawnCmd("/bin/bash", {"-c", bindDirs.value(dirItem)}, cmdLog, err)) {
            err = "ghost install bind error cmd : " + bindDirs.value(dirItem) + " err : " + err;
            return false;
        }
    }

    return true;
}

bool UImgOperate::updateFstab(QString &err)
{
    QString infoFilePath = TmpFileDir + "/" + InfoFileName;
    QString destPath = TargetName + "/etc/fstab";

    QJsonDocument infoDoc;
    if (!Utils::readJsonFile(infoFilePath, infoDoc)) {
        err = "updateFstab read " + infoFilePath + " failed !";
        return false;
    }

    // 获取分区挂载情况和uuid
    QString devicesWithF = " -f " + m_destDevices.join(" -f ");
    QString cmd = QString("lsblk %1 -Jlpo NAME,UUID,MOUNTPOINT,FSTYPE").arg(devicesWithF);
    QJsonArray deviceUuidArray;
    if (!Utils::getLsblkJsonReturn(cmd, deviceUuidArray, err)) {
        err = "updateFstab filed err : " + err;
        return false;
    }

    QMap<QString, QJsonObject> uuidMap;
    for (int i = 0; i < deviceUuidArray.size(); ++i) {
        QJsonObject deviceUuidObj = deviceUuidArray.at(i).toObject();
        if (deviceUuidObj.value("fstype").toString().compare("swap")) {
            if (deviceUuidObj.value("fstype").toString().isEmpty()) {
                continue;
            }
            QString uuidMountPoint = deviceUuidObj.value("mountpoint").toString();
            uuidMountPoint.replace(TargetName, "/");
            uuidMountPoint.replace("//", "/");
            uuidMap[uuidMountPoint] = deviceUuidObj;
        } else {
            uuidMap["none"] = deviceUuidObj;
        }
    }

    QJsonArray bindInfoArray = infoDoc.object().value("fstab").toArray();
    QStringList fstabOutList;
    for (int i = 0; i < bindInfoArray.size(); i++) {
        QJsonObject bindInfoObjItem = bindInfoArray.at(i).toObject();
        QString fstabDeviceValue = "";
        if (bindInfoObjItem.value("options").toString().contains("bind")) {
            fstabDeviceValue = bindInfoObjItem.value("device").toString();
        } else {
            QString bindInfoMountPoint = bindInfoObjItem.value("mountPoint").toString();
            fstabOutList.append(QString("# %1").arg(uuidMap.value(bindInfoMountPoint).value("name").toString()));
            fstabDeviceValue = "UUID=" + uuidMap.value(bindInfoMountPoint).value("uuid").toString();
        }

        fstabOutList.append(QString("%1 %2 %3 %4 %5 %6")
                                    .arg(fstabDeviceValue)
                                    .arg(bindInfoObjItem.value("mountPoint").toString())
                                    .arg(bindInfoObjItem.value("type").toString())
                                    .arg(bindInfoObjItem.value("options").toString())
                                    .arg(bindInfoObjItem.value("dump").toString())
                                    .arg(bindInfoObjItem.value("pass").toString()));
        fstabOutList.append(" ");
    }

    QString fstabStr = fstabOutList.join("\n");

    QFile fstabFile(destPath);
    if (!fstabFile.open(QIODevice::Text | QIODevice::WriteOnly)) {
        err = "updateFstab write " + destPath + " failed !";
        return false;
    }
    fstabFile.write(fstabStr.toStdString().c_str());
    fstabFile.close();

    //更新SystemUpgradeMountPoint中/etc/fstab/中的信息
    if (infoDoc.object().value("systemUpgradeInfo").toObject().value("isSystemUpgrade").toBool()) {
        QStringList fstabOutListB = fstabOutList;
        QString rootNameA = uuidMap.value("/").value("name").toString();
        QString rootUuidA = uuidMap.value("/").value("uuid").toString();
        QString rootNameB = uuidMap.value(SystemUpgradeMountPoint).value("name").toString();
        QString rootUuidB = uuidMap.value(SystemUpgradeMountPoint).value("uuid").toString();

        for (int i = 0; i < fstabOutListB.size(); i++) {
            if (fstabOutListB.at(i).contains(rootNameA)) {
                fstabOutListB[i].replace(rootNameA, rootNameB);
            }
            if (fstabOutListB.at(i).contains(rootUuidA)) {
                fstabOutListB[i].replace(rootUuidA, rootUuidB);
            }
        }

        QString fstabStrB = fstabOutListB.join("\n");

        QFile fstabFileB(TargetName + SystemUpgradeMountPoint + "/etc/fstab");
        if (!fstabFileB.open(QIODevice::Text | QIODevice::WriteOnly)) {
            err = "updateFstab write " + TargetName + SystemUpgradeMountPoint + "/etc/fstab failed !";
            return false;
        }
        fstabFileB.write(fstabStrB.toStdString().c_str());
        fstabFileB.close();
    }

    return true;
}

bool UImgOperate::updateGrub(QString &err)
{
    QString infoFilePath = TmpFileDir + "/" + InfoFileName;
    QJsonDocument infoDoc;
    if (!Utils::readJsonFile(infoFilePath, infoDoc)) {
        err = "updateGrub read " + infoFilePath + " failed !";
        return false;
    }

    // 根据引导模式适配对应的操作
    bool installBootLoaderResult = true;
    bool isEFI = infoDoc.object().value("isEFI").toBool();
    QString platformStr = infoDoc.object().value("platform").toString();
    int currentPlatformType = Utils::currentPlatformByString(platformStr);

    switch (currentPlatformType) {
        case PlatformType::X86_64:
        case PlatformType::I386:
        case PlatformType::I686:
        case PlatformType::AMD64:
        case PlatformType::X86:
            installBootLoaderResult = isEFI ? installBootLoader_x86_efi(err) : installBootLoader_x86_legacy(err);
            break;
        case PlatformType::AARCH64:
            installBootLoaderResult = isEFI ? installBootLoader_arm_efi(err) : true;
            break;
        case PlatformType::UNKNOW:
            err = "updateGrub not support platform type";
            installBootLoaderResult = false;
            break;
        default: /*PlatformType::SW_64、PlatformType::MIPS64、PlatformType::LOONGARCH64*/
            break;
    }

    if (!installBootLoaderResult) {
       return false;
    }

    // 更新/target/etc/default/grub.d/11_deepin_ab_recovery.cfg中记录的升级前的根的uuid和路径信息
    if (infoDoc.object().value("systemUpgradeInfo").toObject().value("isSystemUpgrade").toBool()) {
        QJsonArray partInfo;
        err = "";
        if (!Utils::getLsblkJsonReturn("lsblk -lfpJ -o NAME,UUID,MOUNTPOINT", partInfo, err)) {
            return false;
        }

        for (int i = 0; i < partInfo.size(); ++i) {
            QJsonObject infoObject = partInfo.at(i).toObject();
            QString mountPoint = infoObject.value("mountpoint").toString();
            QString rootPath   = infoObject.value("name").toString();
            QString rootUuid   = infoObject.value("uuid").toString();
            if (!mountPoint.compare(TargetName + SystemUpgradeMountPoint)) {
                QString oldRootPath = infoDoc.object().value("systemUpgradeInfo").toObject().value("rootPath").toString();
                QString oldRootUuid = infoDoc.object().value("systemUpgradeInfo").toObject().value("rootUuid").toString();
                QString cfgFilePath = TargetName + "/etc/default/grub.d/11_deepin_ab_recovery.cfg";
                QFile file(cfgFilePath);
                if (!file.open(QIODevice::Text | QIODevice::ReadOnly)) {
                    err = "read " + cfgFilePath + " failed !";
                    return false;
                }
                QString data = file.readAll();
                file.close();

                data.replace(oldRootPath, rootPath);
                data.replace(oldRootUuid, rootUuid);

                file.setFileName(cfgFilePath);
                if (!file.open(QIODevice::Text | QIODevice::WriteOnly)) {
                    err = "write " + cfgFilePath + " failed !";
                    return false;
                }
                file.write(data.toLocal8Bit());
                file.close();
                break;
            }
        }
    }

    return setupBootLoader(err);
}

bool UImgOperate::installBootLoader_x86_efi(QString &err)
{
    int platformSize = 0;
    if (!getPlatformSize(platformSize, err)) {
        return false;
    }

    bool isCommunity = false;
    QString bootloaderId = "uos";
    static QString osEditionType = Utils::getOSEditionType();
    if ("Community" == osEditionType) {
        bootloaderId = "deepin";
        isCommunity = true;
    }

    QString cmd = "";
    // 基础启动项,默认UOS
    if (platformSize == 32) {
        cmd = QString("chroot %1 grub-install --target=i386-efi --efi-directory=/boot/efi --bootloader-id=\"%2\" --recheck")
                .arg(TargetName).arg(bootloaderId);
    } else if (platformSize == 64) {
        cmd = QString("chroot %1 grub-install --target=x86_64-efi --uefi-secure-boot --efi-directory=/boot/efi --bootloader-id=\"%2\" --recheck")
                .arg(TargetName).arg(bootloaderId);
    } else {
        err = "ghost install installBootLoader_x86_efi error no support size";
        return false;
    }

    QString cmdLog = "";
    err = "";
    if (!Process::spawnCmd("/bin/bash",{"-c", cmd}, cmdLog, err)) {
        err = "ghost install installBootLoader_x86_efi error cmd : " + cmd + " err : " + err;
        return false;
    }

    if (isCommunity) {
        QString deepinSrc = QString("%1/boot/efi/EFI/%2/grub*").arg(TargetName).arg(bootloaderId);
        QString deepinDest = QString("%1/boot/efi/EFI/boot/").arg(TargetName);
        if (!copyGrubFile(deepinSrc, deepinDest, err)) {
            qWarning()<<Q_FUNC_INFO<<"copyGrubFile failed, deepinSrc = "<<deepinSrc<<", err = "<<err;
            return false;
        }
    } else {
        // 拷贝更新后的grub文件目录到boot目录
        if (!copyGrubFile(QString("%1/boot/efi/EFI/UOS/grub*").arg(TargetName), QString("%1/boot/efi/EFI/boot/").arg(TargetName), err)) {
            qWarning()<<Q_FUNC_INFO<<"copyGrubFile failed, TargetName = "<<TargetName<<", err = "<<err;
            return false;
        }
    }

    return true;
}

bool UImgOperate::installBootLoader_x86_legacy(QString &err)
{
    QString cmd = QString("chroot %1 grub-install --no-floppy %2 --target=i386-pc --force 2>/dev/null").arg(TargetName).arg(m_destDevices.at(0));
    QString cmdLog = "";
    if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, err)) {
        err = "ghost install installBootLoader_x86_legacy error cmd : " + cmd + " err : " + err;
        return false;
    }
    return true;
}

bool UImgOperate::installBootLoader_arm_efi(QString &err)
{
    int platformSize = 0;
    if (!getPlatformSize(platformSize, err)) {
        return false;
    }

    QString cmd = "";
    if (platformSize == 32) {
        cmd = QString("chroot %1 grub-install --target=arm-efi --efi-directory=/boot/efi --bootloader-id=\"UOS\" --recheck").arg(TargetName);
    } else if (platformSize == 64) {
        cmd = QString("chroot %1 grub-install --target=arm64-efi --uefi-secure-boot --efi-directory=/boot/efi --bootloader-id=\"UOS\" --recheck").arg(TargetName);
    } else {
        err = "ghost install installBootLoader_arm_efi error no support size";
        return false;
    }

    QString cmdLog = "";
    err = "";
    if (!Process::spawnCmd("/bin/bash",{"-c", cmd}, cmdLog, err)) {
        err = "ghost install installBootLoader_arm_efi error cmd : " + cmd + " err : " + err;
        return false;
    }

    // 拷贝更新后的grub文件目录到boot目录
    if (!copyGrubFile(QString("%1/boot/efi/EFI/UOS/grub*").arg(TargetName), QString("%1/boot/efi/EFI/boot/").arg(TargetName), err)) {
        return false;
    }

    return true;
}

bool UImgOperate::setupBootLoader(QString &err)
{
    QString cmd = QString("chroot %1 /usr/sbin/update-grub").arg(TargetName);
    QString cmdLog = "";
    if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, err)) {
        err = "ghost install setupBootLoader error cmd : " + cmd + " err : " + err;
        return false;
    }

    cmd = QString("chroot %1 update-initramfs -u").arg(TargetName);
    cmdLog = "";
    err = "";
    if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, err)) {
        err = "ghost install setupBootLoader error cmd : " + cmd + " err : " + err;
        return false;
    }

    return true;
}

int UImgOperate::getOSMajorVersion(const QString &rootmnt)
{
    QString osVerPath = "/etc/os-version";
    if ("/" != rootmnt) {
        osVerPath = rootmnt + "/etc/os-version";
    }

    int osVersion = 0;
    QSettings versionFile(osVerPath, QSettings::IniFormat);
    versionFile.beginGroup("Version");
    osVersion = versionFile.value("MajorVersion").toInt();
    versionFile.endGroup();

    return osVersion;
}

bool UImgOperate::updateUdisksRules(const QString &rootmnt, const QString &fstabPath, const QString &rulePath)
{
    FSTabInfoList fstabList = FSTab::getFSTabFromFile(fstabPath);
    QStringList sysDeviceUuidList = FSTab::getUuidListFromFstab(fstabList);

    DeviceInfoList diskList;
    DeviceInfoList cryptDevList;
    DeviceInfoList noCryptDevList;
    DeviceInfoList partitionList;
    DeviceInfoList lvmList;
    Device::getAllTypeDevice(diskList, cryptDevList, noCryptDevList, partitionList, lvmList);

    int osMajorVer = UImgOperate::getOSMajorVersion(rootmnt);
    QStringList cryptLuksUuidList;
    QMap<QString, QString> sysUuidMountPointMap; // QMap<uuid, mountPoint>

    for (auto &diskDev : diskList) {
        for (auto &cryptDev : cryptDevList) {
            if (!(cryptDev->type == "part" && cryptDev->fsType == "luks" && cryptDev->pkname == diskDev->name)) {
                continue;
            }
            cryptLuksUuidList.append(cryptDev->uuid);

            for (auto &lvm2Dev : cryptDevList) {
                if (!(lvm2Dev->type == "crypt" && lvm2Dev->fsType == "lvm2" && lvm2Dev->pkname == cryptDev->kname)) {
                    continue;
                }

                for (auto &lvmMember : lvmList) {
                    if (lvmMember->pkname == lvm2Dev->kname) {
                        if (sysDeviceUuidList.contains(lvmMember->uuid)) {
                            sysUuidMountPointMap.insert(lvmMember->uuid, lvmMember->mountPoint);
                        } else {
                            if (OsMajorVersionV20 == osMajorVer && lvmMember->mountPoint.isEmpty() &&
                                (lvmMember->label == "Roota" || lvmMember->label == "Rootb")) {
                                sysUuidMountPointMap.insert(lvmMember->uuid, AnotherRootMountPoint);
                            }
                        }
                    }
                }
            }
        }

        for (auto &noCryptDev : noCryptDevList) {
            if (noCryptDev->pkname != diskDev->name) {
                continue;
            }

            for (auto &lvmMember : lvmList) {
                if (lvmMember->pkname == noCryptDev->kname) {
                    if (sysDeviceUuidList.contains(lvmMember->uuid)) {
                        sysUuidMountPointMap.insert(lvmMember->uuid, lvmMember->mountPoint);
                    } else {
                        if (OsMajorVersionV20 == osMajorVer && lvmMember->mountPoint.isEmpty() &&
                            (lvmMember->label == "Roota" || lvmMember->label == "Rootb")) {
                            sysUuidMountPointMap.insert(lvmMember->uuid, AnotherRootMountPoint);
                        }
                    }
                }
            }
        }

        for (auto &partitionDev : partitionList) {
            if (partitionDev->pkname == diskDev->name) {
                if (sysDeviceUuidList.contains(partitionDev->uuid)) {
                    sysUuidMountPointMap.insert(partitionDev->uuid, partitionDev->mountPoint);
                } else {
                    if (OsMajorVersionV20 == osMajorVer && partitionDev->mountPoint.isEmpty() &&
                        (partitionDev->label == "Roota" || partitionDev->label == "Rootb")) {
                        sysUuidMountPointMap.insert(partitionDev->uuid, AnotherRootMountPoint);
                    }
                }
            }
        }
    }

    QStringList ruleList = UImgOperate::getUdiskRules(rootmnt, cryptLuksUuidList, sysUuidMountPointMap);

    QFile writeRuleFile(rulePath);
    if (!writeRuleFile.open(QIODevice::Truncate | QIODevice::Text | QIODevice::WriteOnly)) {
        qCritical()<<"updateUdisksRules, writeRuleFile open failed, rulePath = "<<rulePath;
        return false;
    }

    for (auto &item : ruleList) {
        writeRuleFile.write(item.toStdString().c_str());
    }
    writeRuleFile.close();

    return true;
}

QStringList UImgOperate::getUdiskRules(const QString &rootmnt, const QStringList &cryptLuksUuidList,
    const QMap<QString, QString> &sysUuidMountPointMap)
{
    QString efiUuid;
    QString bootUuid;
    QString swapUuid;
    QString rootbUuid;
    QString recoveryUuid;
    QString rootPath = "/";
    if ("/" != rootmnt) {
        rootPath = rootmnt + "/";
    }
    QString efiMountPoint = rootPath + "boot/efi";
    QString bootMountPoint = rootPath + "boot";

    for (auto &uuid : sysUuidMountPointMap.keys()) {
        QString mountPoint = sysUuidMountPointMap[uuid];
        if (mountPoint.isEmpty() || "/" == mountPoint) {
            continue;
        }
        if (efiMountPoint == mountPoint) {
            efiUuid = uuid;
        } else if (bootMountPoint == mountPoint) {
            bootUuid = uuid;
        } else if ("[SWAP]" == mountPoint) {
            swapUuid = uuid;
        } else if (AnotherRootMountPoint == mountPoint) {
            rootbUuid = uuid;
        } else {
            QString systemDimPath = mountPoint + "/backup/system.dim";
            QDir recoveryDir;
            if (recoveryDir.exists(systemDimPath)) {
                recoveryUuid = uuid;
                continue;
            }
        }
    }

    QStringList ruleList;
    if (!efiUuid.isEmpty()) {
        QString hideEfi = "# hide efi \n";
        QString efiItem = QString("ENV{ID_FS_UUID}==\"%1\", ENV{UDISKS_IGNORE}=\"1\"\n").arg(efiUuid);
        ruleList.append(hideEfi);
        ruleList.append(efiItem);
    }

    if (!bootUuid.isEmpty()) {
        QString hideBoot = "# hide boot \n";
        QString bootItem = QString("ENV{ID_FS_UUID}==\"%1\", ENV{UDISKS_IGNORE}=\"1\"\n").arg(bootUuid);
        ruleList.append(hideBoot);
        ruleList.append(bootItem);
    }

    if (!rootbUuid.isEmpty()) {
        QString hideRootb = "# hide rootb \n";
        QString rootbItem = QString("ENV{ID_FS_UUID}==\"%1\", ENV{UDISKS_IGNORE}=\"1\"\n").arg(rootbUuid);
        ruleList.append(hideRootb);
        ruleList.append(rootbItem);
    }

    if (!swapUuid.isEmpty()) {
        QString hideSwap = "# hide swap \n";
        QString swapItem = QString("ENV{ENV{ID_FS_TYPE}==\"swap\", ENV{UDISKS_IGNORE}=\"1\"\n");
        ruleList.append(hideSwap);
        ruleList.append(swapItem);
    }

    if (!recoveryUuid.isEmpty()) {
        QString hideRecovery = "# hide recovery \n";
        QString recoveryItem = QString("ENV{ID_FS_UUID}==\"%1\", ENV{UDISKS_IGNORE}=\"1\"\n").arg(recoveryUuid);
        ruleList.append(hideRecovery);
        ruleList.append(recoveryItem);
    }

    if (!cryptLuksUuidList.isEmpty()) {
        QString hideCryptoLuks = "# hide crypto_luks \n";
        ruleList.append(hideCryptoLuks);

        for (auto &cryptUuid : cryptLuksUuidList) {
            QString cryptItem = QString("ENV{ID_FS_UUID}==\"%1\", ENV{UDISKS_IGNORE}=\"1\"\n").arg(cryptUuid);
            ruleList.append(cryptItem);
        }
    }

    return ruleList;
}

bool UImgOperate::backUpConfFiles(QString &err)
{
    QString infoFilePath = TmpFileDir + "/" + InfoFileName;
    QJsonDocument infoDoc;
    if (!Utils::readJsonFile(infoFilePath, infoDoc)) {
        err = "updateGrub read " + infoFilePath + " failed !";
        return false;
    }

    QJsonArray deviceInfos;
    err = "";
    if (!Utils::getLsblkJsonReturn("lsblk -lfpJ -o UUID,MOUNTPOINT", deviceInfos, err)) {
        return false;
    }

    QString ruleFilePath = "/etc/udev/rules.d/80-udisks-installer.rules";
    QString fstabPath = TargetName + "/etc/fstab";
    if (!UImgOperate::updateUdisksRules(TargetName, fstabPath, TargetName + ruleFilePath)) {
        //TODO: v20 hide rootb
        qWarning()<<"updateUdisksRules failed";
    }
    // 还原.back文件到/target/etc/udev/rules.d/80-udisks-installer.rules
    // replaceRootUuidContent(TargetName + ruleFilePath + ".back",
    //                        TargetName + ruleFilePath,
    //                        deviceInfos);

    QString abRecoveryConfFilePath = "/etc/deepin/ab-recovery.json";
    // 还原.back文件到/target/etc/deepin/ab-recovery.json
    replaceRootUuidContent(TargetName + abRecoveryConfFilePath + ".back",
                           TargetName + abRecoveryConfFilePath,
                           deviceInfos);

    if (infoDoc.object().value("systemUpgradeInfo").toObject().value("isSystemUpgrade").toBool()) {
        // 还原.back文件到/SystemUpgradeMountPoint/etc/udev/rules.d/80-udisks-installer.rules
        replaceRootUuidContent(TargetName + SystemUpgradeMountPoint + ruleFilePath + ".back",
                               TargetName + SystemUpgradeMountPoint + ruleFilePath,
                               deviceInfos);

        // 还原.back文件到/SystemUpgradeMountPoint/etc/deepin/ab-recovery.json
        replaceRootUuidContent(TargetName + SystemUpgradeMountPoint + abRecoveryConfFilePath + ".back",
                               TargetName + SystemUpgradeMountPoint + abRecoveryConfFilePath,
                               deviceInfos);
    }

    // 删除SystemUpgradeMountPoint目录
    QDir deleteDir(TargetName + SystemUpgradeMountPoint);
    if (deleteDir.exists()) {
        QString cmd = QString("umount %1").arg(TargetName + SystemUpgradeMountPoint);
        QString cmdLog = "";
        err = "";
        if (Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, err)) {
            cmd = QString("rm -rf %1").arg(TargetName + SystemUpgradeMountPoint);
            cmdLog = "";
            err = "";
            if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, err)) {
                qInfo()<<"rm failed, cmd = "<<cmd<<", err = "<<err;
            }
        }
    }
    return true;
}

bool UImgOperate::checkImgFileInDisk(const QString &imgFile, const QStringList &devices, QString &err)
{
    QJsonObject destInfo;
    if (!Utils::getDestPartInfoByDir(imgFile, destInfo, err)) {
        return false;
    }

    QString diskName = destInfo.value("name").toString();
    if (std::find_if(devices.begin(), devices.end(), [=](QString devItem) { return diskName.contains(devItem); }) != devices.end()) {
        return false;
    }

    return true;
}

bool UImgOperate::checkDiskSize(const qint64 &deviceSize, const QStringList &devices, QString &err)
{
    m_destDevices.clear();
    qint64 devToUseSize = 0;
    Device devToUse;
    DeviceInfoList devicesToUse = devToUse.getDeviceByLsblk();
    DeviceInfoList devicesToUseList;
    qInfo()<<"checkDiskSize:, deviceSize = "<<deviceSize<<", devices = "<<devices;
    for (DeviceInfoPtr deviceItem : devicesToUse) {
        if (devices.contains(deviceItem->path)) {
            devToUseSize = devToUseSize + deviceItem->sizeBytes;
            devicesToUseList.append(deviceItem);
            qInfo()<<"checkDiskSize:, devicesToUseList deviceItem->path = "<<deviceItem->path;
        }
    }
    if (devicesToUseList.size() > 0) {
        std::sort(devicesToUseList.begin(), devicesToUseList.end(), [](DeviceInfoPtr pair1, DeviceInfoPtr pair2){ return pair1->sizeBytes < pair2->sizeBytes; });
    } else {
        err = "checkDiskSize devicesToUseList is empty !";
        return false;
    }

    // 带阈值校验总空间大小
    if (devToUseSize < deviceSize) {
        qint64 subSize = deviceSize - devToUseSize;
        qInfo()<<"checkDiskSize:, devToUseSize = "<<devToUseSize<<", deviceSize = "<<deviceSize<<", subSize = "<<subSize;
        if (subSize > RangeSpace) {
            err = "checkDiskSize out of RangeSpace !";
            return false;
        }
    }

    qInfo()<<"checkDiskSize:, devicesToUseList.size() = "<<devicesToUseList.size();

    if (devicesToUseList.size() > 1) {
        // 检测是否存在有分区两个设备都放不下的情况，还要考虑lvm分区
        QJsonDocument policyDoc;
        if (!Utils::readJsonFile(SquashFileMountDir + "/" + PolicyFileName, policyDoc)) {
            err = "checkDiskSize read " + SquashFileMountDir + "/" + PolicyFileName + " failed !";
            return false;
        }
        typedef QPair<QString, qint64> itemPair;
        QList<itemPair> partsSizeList;
        QJsonArray policyArray = policyDoc.array();
        for (int i = 0; i < policyArray.size(); i++) {
            QJsonObject policyObjItem = policyArray.at(i).toObject();
            if (!policyObjItem.value("type").toString().compare("partition")) {
                QString partId =  policyObjItem.value("id").toString();
                qint64 partSize = policyObjItem.value("size").toVariant().toLongLong();
                partsSizeList.append(itemPair(partId, partSize));
            }
        }
        std::sort(partsSizeList.begin(), partsSizeList.end(), [](itemPair pair1, itemPair pair2){ return pair1.second < pair2.second; });

        // 所选设备能不能存下所有的分区
        qint64 firstDevTotalPartSize = 0;
        qint64 secondDevTotalPartSize = 0;
        for (itemPair partItem: partsSizeList) {
            firstDevTotalPartSize += partItem.second;
            if (firstDevTotalPartSize > devicesToUseList.first()->sizeBytes) {
                secondDevTotalPartSize += partItem.second;
                if (secondDevTotalPartSize > devicesToUseList.last()->sizeBytes) {
                    return false;
                }
            }
        }

        // 检测是否存在一个设备满足所有的空间要求
        // auto resultIter = std::find_if(devicesToUseList.begin(), devicesToUseList.end(), [=](DeviceInfoPtr devItem){ return devItem->sizeBytes > deviceSize; });
        // if (resultIter != devicesToUseList.end()) {
        //     m_destDevices.append((*resultIter)->path);
        // }

        // 不存在则把所有设备都算上
        if (m_destDevices.size() == 0) {
            for (DeviceInfoPtr devItem : devicesToUseList) {
                m_destDevices.append(devItem->path);
            }
        }

    } else {
        m_destDevices.append(devicesToUseList.first()->path);
    }

    return true;
}


bool UImgOperate::getDeletePartList(QJsonArray &partList, QString &err)
{
    // 获取本机的lvm分区信息设置分区删除项
    if (!Device::exportPartitionInfoByLsblk(TmpFileDir + TmpFileDir + PolicyFileName, m_destDevices, err)) {
        return false;
    }

    QJsonDocument policyDoc;
    if (!Utils::readJsonFile(TmpFileDir + TmpFileDir + PolicyFileName, policyDoc)) {
        err = "getDeletePartList read " + TmpFileDir + TmpFileDir + PolicyFileName + " failed !";
        return false;
    }

    QJsonArray policyArray = policyDoc.array();
    for (int i = 0; i < policyArray.size(); i++) {
        QJsonObject policyObjItem = policyArray.at(i).toObject();
        if (!policyObjItem.value("type").toString().compare("VG")) {
            QJsonObject policyObjOutItem;
            policyObjOutItem.insert("id", policyObjItem.value("id"));
            policyObjOutItem.insert("name", policyObjItem.value("name"));
            policyObjOutItem.insert("operate", "delete");
            policyObjOutItem.insert("pv", policyObjItem.value("pv"));
            policyObjOutItem.insert("size", policyObjItem.value("size"));
            policyObjOutItem.insert("type", policyObjItem.value("type"));
            partList.append(policyObjOutItem);
        }

        if (!policyObjItem.value("type").toString().compare("LVM")) {
            QJsonObject policyObjOutItem;
            policyObjOutItem.insert("id", policyObjItem.value("id"));
            policyObjOutItem.insert("filesystem", policyObjItem.value("filesystem"));
            policyObjOutItem.insert("label", policyObjItem.value("label"));
            policyObjOutItem.insert("mountPoint", policyObjItem.value("mountPoint"));
            policyObjOutItem.insert("operate", "delete");
            policyObjOutItem.insert("size", policyObjItem.value("size"));
            policyObjOutItem.insert("type", policyObjItem.value("type"));
            policyObjOutItem.insert("vg", policyObjItem.value("vg"));
            partList.append(policyObjOutItem);
        }
    }

    return true;
}

bool UImgOperate::updatePartStartEndPoint(QJsonObject &partObj, const QJsonObject &prePartObj,
    const qint64 &partTotalBytes, const qint64 &devBytes, const qint64 &sectorSize, bool isSameDisk)
{
    // TODO:需要考虑不同扇区大小的分区对齐问题

    qint64 subSectorBytes = 0;
    qint64 partSize = partObj.value("size").toVariant().toLongLong() * MiB;
    qint64 startPoint = partObj.value("startPoint").toVariant().toLongLong();
    qint64 endPoint = partObj.value("endPoint").toVariant().toLongLong();
    qint64 newStartPoint = 0;
    qint64 newEndPoint = 0;

    QString partObjStr = Utils::JsonToQString(partObj);
    QString prePartObjStr = Utils::JsonToQString(const_cast<QJsonObject&> (prePartObj));
    qInfo()<<"updatePartStartEndPoint: partTotalBytes = "<<partTotalBytes<<", devBytes = "<<devBytes
           <<", sectorSize = "<<sectorSize;
    qInfo()<<"updatePartStartEndPoint: partObjStr = "<<partObjStr;
    qInfo()<<"updatePartStartEndPoint: prePartObjStr = "<<prePartObjStr;

    if (partTotalBytes > devBytes) {
        subSectorBytes = partTotalBytes - devBytes;
        if (subSectorBytes <= RangeSpace) {
            newStartPoint = startPoint;
            qint64 sectorsNeed = subSectorBytes / sectorSize;
            newEndPoint = endPoint - (sectorsNeed * sectorSize);
            qInfo()<<"updatePartStartEndPoint:1 newEndPoint = "<<newEndPoint<<", newStartPoint = "<<newStartPoint
                   <<", subSectorBytes = "<<subSectorBytes<<", endPoint = "<<endPoint;
        } else {
            qInfo()<<"updatePartStartEndPoint: subSectorBytes = "<<subSectorBytes<<", partTotalBytes = "<<partTotalBytes
                <<", devBytes = "<<devBytes<<", isSameDisk = "<<isSameDisk;
            if (isSameDisk) {
                return false;
            }

            newStartPoint = startPoint;
            newEndPoint = endPoint;
            qInfo()<<"updatePartStartEndPoint: newStartPoint = "<<newStartPoint<<", newEndPoint = "<<newEndPoint
                   <<", isSameDisk = "<<isSameDisk;
        }
    } else {
        subSectorBytes = devBytes - partTotalBytes;
        if (subSectorBytes <= RangeSpace) {
            newStartPoint = startPoint;
            // qint64 sectorsNeed = subSectorBytes / sectorSize;
            // newEndPoint = endPoint + (sectorsNeed * sectorSize);
            newEndPoint = endPoint;
            qInfo()<<"updatePartStartEndPoint:2 newEndPoint = "<<newEndPoint<<", newStartPoint = "<<newStartPoint
                   <<", subSectorBytes = "<<subSectorBytes<<", endPoint = "<<endPoint;
        } else {
            newStartPoint = startPoint;
            newEndPoint = endPoint;
            qInfo()<<"updatePartStartEndPoint:3 newEndPoint = "<<newEndPoint<<", newStartPoint = "<<newStartPoint;
        }
    }

    // 判断当前分区的起点和上一个分区的起点的间隙，小于视为存在空间重叠，调整起点，紧接上一个分区的终点，根据起止点调整分区大小
    if (!prePartObj.isEmpty()) {
        qint64 preStartPoint = prePartObj.value("startPoint").toVariant().toLongLong();
        qint64 preEndPoint = prePartObj.value("endPoint").toVariant().toLongLong();
        qInfo()<<"updatePartStartEndPoint:4 preStartPoint = "<<preStartPoint<<", preEndPoint = "<<preEndPoint;
        if (isSameDisk && newStartPoint < preStartPoint) {
            newStartPoint = preEndPoint + 1;
            // 防止按边界调整后newEndPoint值被修改
            if (newEndPoint == endPoint) {
                newEndPoint = newStartPoint - 1 - (partSize / sectorSize);
                qInfo()<<"updatePartStartEndPoint:5 newEndPoint = "<<newEndPoint<<", endPoint = "<<endPoint;
            } else {
                partSize = ((newEndPoint + 1 - startPoint) * sectorSize) / MiB;
                partObj["size"] = QString::number(partSize);
            }
        }
    }

    partObj["startPoint"] = QString::number(newStartPoint);
    partObj["endPoint"] = QString::number(newEndPoint);

    return true;
}

bool UImgOperate::getPlatformSize(int &size, QString &err)
{
    // 获取总线值
    QString cmd = "getconf LONG_BIT";
    QString cmdLog = "";
    err = "";
    if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, err)) {
        err = "getPlatformSize error cmd : " + cmd + " err : " + err;
        return false;
    }

    size = cmdLog.toInt();

    return true;
}

bool UImgOperate::copyGrubFile(const QString &src, const QString &dest, QString &err)
{
    QString cmd = QString("cp -vf %1 %2").arg(src).arg(dest);
    QString cmdLog = "";
    err = "";
    if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, err)) {
        err = "copyGrubFile error cmd : " + cmd + " err : " + err;
        return false;
    }

    return true;
}

void UImgOperate::replaceRootUuidContent(const QString &srcFilePath, const QString &destFilePath, const QJsonArray &devInfos)
{
    QFile ruleFileBack(srcFilePath);
    if (!ruleFileBack.open(QIODevice::Text | QIODevice::ReadOnly)) {
        qWarning()<<"replaceRootUuidContent, open srcFilePath = "<<srcFilePath<<", failed";
        return;
    }
    QString data = ruleFileBack.readAll();
    ruleFileBack.close();
    qInfo()<<"replaceRootUuidContent, srcFilePath = "<<srcFilePath<<", destFilePath = "<<destFilePath<<" data = "<<data;

    for (int i = 0; i < devInfos.size(); i++) {
        QJsonObject infoItem = devInfos.at(i).toObject();
        QString partUuid = infoItem.value("uuid").toString();
        QString partMountPoint = infoItem.value("mountpoint").toString();
        if (partUuid.isEmpty() || partMountPoint.isEmpty()) {
            continue;
        }
        qInfo()<<"replaceRootUuidContent, partUuid = "<<partUuid<<", partMountPoint = "<<partMountPoint<<
            ", TargetName = "<<TargetName;
        if (!partMountPoint.compare(TargetName)) {
            data.replace(TargetName, partUuid);
        } else {
            partMountPoint.replace(TargetName, "/");
            partMountPoint.replace("//", "/");
            data.replace(partMountPoint, partUuid);
        }
    }

    QFile ruleFile(destFilePath);
    if (!ruleFile.open(QIODevice::Text | QIODevice::WriteOnly)) {
        qWarning()<<"replaceRootUuidContent, open destFilePath = "<<destFilePath<<", failed";
        return;
    }
    ruleFile.write(data.toLocal8Bit());
    ruleFile.close();
    qInfo()<<"replaceRootUuidContent, dest data = "<<data;

    QFile::remove(srcFilePath);
}

bool UImgOperate::renameVgName(const QString &partitionFile, const QStringList &sysVgNameList)
{
    QJsonDocument policyDoc;
    if (!Utils::readJsonFile(partitionFile, policyDoc)) {
        qCritical()<<"UImgOperate::renameVgName failed, partitionFile = "<<partitionFile;
        return false;
    }

    QMap<QString, QString> vgMap;
    QJsonArray policyArray = policyDoc.array();
    int policyArraySize = policyArray.size();
    for (int i = 0; i < policyArraySize; ++i) {
        QJsonObject policyObjItem = policyArray.at(i).toObject();
        QString type = policyObjItem.value("type").toString();
        QString operate = policyObjItem.value("operate").toString();
        if (operate == "new" && type == "VG") {
            QString vgName = policyObjItem.value("name").toString();
            vgMap.insert(vgName, vgName);
        }
    }

    if (!Device::renameVG(sysVgNameList, vgMap)) {
        // vg name not conflict
        return true;
    }

    // update vg name
    QJsonArray newPolicyArray;
    for (int i = 0; i < policyArraySize; ++i) {
        QJsonObject policyObjItem = policyArray.at(i).toObject();
        QString operate = policyObjItem.value("operate").toString();
        if (operate == "new") {
            QString type = policyObjItem.value("type").toString();
            if (type == "VG") {
                QString vgName = policyObjItem.value("name").toString();
                policyObjItem.remove("name");
                policyObjItem.insert("name", vgMap[vgName]);
            } else if (type == "LVM") {
                QString vgName = policyObjItem.value("vg").toString();
                policyObjItem.remove("vg");
                policyObjItem.insert("vg", vgMap[vgName]);
            }
        }

        newPolicyArray.append(policyObjItem);
    }

    QJsonDocument policyDocOut(newPolicyArray);
    QFile file(partitionFile);
    if (!file.open(QIODevice::Text | QIODevice::WriteOnly | QIODevice::Truncate)) {
       qCritical()<<"UImgOperate::renameVgName open failed, partitionFile = "<<partitionFile;
        return false;
    }
    file.write(policyDocOut.toJson());
    file.close();

    qInfo()<<"newPolicyArray: "<<newPolicyArray;

    return true;
}

QJsonObject GhostCheckItem::marshal()
{
    QJsonObject jsonObject;
    jsonObject.insert("diskNum", diskNum);
    jsonObject.insert("diskSize", QString("%1").arg(diskSize));
    jsonObject.insert("bootLoader", bootLoader);
    jsonObject.insert("arch", arch);
    jsonObject.insert("displayCard", displayCard);
    jsonObject.insert("cpu", cpu);
    jsonObject.insert("readUimg", readUimg);
    jsonObject.insert("readFile", readFile);
    jsonObject.insert("installPos", installPos);

    return jsonObject;
}

void GhostCheckItem::unmarshal(const QJsonObject &jsonObject)
{
    bool ok = false;
    diskNum = jsonObject.value("diskNum").toInt(-1);
    diskSize = jsonObject.value("diskSize").toString().toULongLong(&ok);
    if (!ok) {
        diskSize = 0;
    }
    bootLoader = jsonObject.value("bootLoader").toString();
    arch = jsonObject.value("arch").toString();
    displayCard = jsonObject.value("displayCard").toString();
    cpu = jsonObject.value("cpu").toString();
    readUimg = jsonObject.value("readUimg").toBool();
    readFile = jsonObject.value("readFile").toBool();
    installPos = jsonObject.value("installPos").toBool();
}

bool UImgOperate::checkDiskSize(QList<GhostDiskInfo>& srcDiskInfoList, const QStringList &devices, GhostCheckItem &src,
                                GhostCheckItem &dest)
{
    // not support hot plug
    static DeviceInfoList allDeviceList = Device::getAllDeviceByLsblk();
    quint64 installDiskTotalSize = 0;
    QList<GhostDiskInfo> destDiskList;
    DeviceInfoList devicesToUseList;
    for (DeviceInfoPtr &deviceItem : allDeviceList) {
        if (devices.contains(deviceItem->path)) {
            installDiskTotalSize += deviceItem->sizeBytes;
            devicesToUseList.append(deviceItem);

            GhostDiskInfo diskInfo;
            Device::fillGhostDiskInfo(deviceItem, diskInfo);
            destDiskList.append(diskInfo);
        }
    }

    int srcDiskNum = srcDiskInfoList.size();
    int destDiskNum = destDiskList.size();
    if (srcDiskNum != destDiskNum) {
        src.diskNum = srcDiskNum;
        dest.diskNum = destDiskNum;
        qCritical()<<"checkDiskSize failed, srcDiskNum = "<<srcDiskNum<<", destDiskNum = "<<destDiskNum;
        return false;
    }

    std::sort(srcDiskInfoList.begin(), srcDiskInfoList.end());
    std::sort(destDiskList.begin(), destDiskList.end());
    quint64 thresholdBytes = 200 * 1024 * 1024; // 200M
    for (int i = 0; i < srcDiskNum; ++i) {
        const GhostDiskInfo &srcDiskInfo = srcDiskInfoList.at(i);
        const GhostDiskInfo &destDiskInfo = destDiskList.at(i);
        if (destDiskInfo.totalSizeBytes < srcDiskInfo.totalSizeBytes) {
            src.diskSize = srcDiskInfo.totalSizeBytes;
            dest.diskSize = destDiskInfo.totalSizeBytes;
            quint64 diffBytes = srcDiskInfo.totalSizeBytes - destDiskInfo.totalSizeBytes;
            quint64 lastPartUsedSize = srcDiskInfo.lastPartitionUsedSizeBytes;
            quint64 srcIdleBytes = 0;
            if (srcDiskInfo.totalSizeBytes <= srcDiskInfo.allocTotalSizeBytes) {
                srcIdleBytes = srcDiskInfo.totalSizeBytes - srcDiskInfo.allocTotalSizeBytes;
            }
            qWarning()<<"checkDiskSize, src diskSize = "<<src.diskSize<<", dest diskSize = "<<dest.diskSize<<
                       ", src devName = "<<srcDiskInfo.deviceName<<", dest devName = "<<destDiskInfo.deviceName<<
                       ", diffBytes = "<<diffBytes<<", srcIdleBytes = "<<srcIdleBytes;

//            if (srcIdleBytes <= diffBytes || diffBytes <= lastPartUsedSize) {
//                qCritical()<<"checkDiskSize failed, lastPartUsedSize = "<<lastPartUsedSize;
//                return false;
//            }
//
//            if (srcDiskInfo.isLvm) {
//                if ((i == srcDiskNum - 1) && diffBytes < lastPartUsedSize + thresholdBytes) {
//                    qCritical()<<"checkDiskSize failed, lvm lastPartUsedSize = "<<lastPartUsedSize;
//                    return false;
//                }
//            }
            return false;
        }
    }

    return true;
}
