/**
 * @license
 * Copyright 2021 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { __decorate, __metadata } from '../../../../tslib/tslib.js';
import { html, LitElement } from '../../../../lit/index.js';
import { property, queryAssignedElements } from '../../../../lit/decorators.js';
import { classMap } from '../../../../lit/directives/class-map.js';
import { ifDefined } from '../../../../lit/directives/if-defined.js';
import { ariaProperty } from '../../decorators/aria-property.js';
/**
 * SegmentedButtonSet is the parent component for two or more
 * `SegmentedButton` components. **Only** `SegmentedButton` components may be
 * used as children.
 * @soyCompatible
 */
export class SegmentedButtonSet extends LitElement {
    constructor() {
        super(...arguments);
        this.multiselect = false;
    }
    getButtonDisabled(index) {
        if (this.indexOutOfBounds(index))
            return false;
        return this.buttons[index].disabled;
    }
    setButtonDisabled(index, disabled) {
        if (this.indexOutOfBounds(index))
            return;
        this.buttons[index].disabled = disabled;
    }
    getButtonSelected(index) {
        if (this.indexOutOfBounds(index))
            return false;
        return this.buttons[index].selected;
    }
    setButtonSelected(index, selected) {
        // Ignore out-of-index values.
        if (this.indexOutOfBounds(index))
            return;
        // Ignore disabled buttons.
        if (this.getButtonDisabled(index))
            return;
        if (this.multiselect) {
            this.buttons[index].selected = selected;
            this.emitSelectionEvent(index);
            return;
        }
        // Single-select segmented buttons are not unselectable.
        if (!selected)
            return;
        this.buttons[index].selected = true;
        this.emitSelectionEvent(index);
        // Deselect all other buttons for single-select.
        for (let i = 0; i < this.buttons.length; i++) {
            if (i === index)
                continue;
            this.buttons[i].selected = false;
        }
    }
    handleSegmentedButtonInteraction(e) {
        const index = this.buttons.indexOf(e.target);
        this.toggleSelection(index);
    }
    toggleSelection(index) {
        if (this.indexOutOfBounds(index))
            return;
        this.setButtonSelected(index, !this.buttons[index].selected);
    }
    indexOutOfBounds(index) {
        return index < 0 || index >= this.buttons.length;
    }
    emitSelectionEvent(index) {
        this.dispatchEvent(new CustomEvent('segmented-button-set-selection', {
            detail: {
                button: this.buttons[index],
                selected: this.buttons[index].selected,
                index,
            },
            bubbles: true,
            composed: true
        }));
    }
    /** @soyTemplate */
    render() {
        return html `
     <span
       role="group"
       @segmented-button-interaction="${this.handleSegmentedButtonInteraction}"
       aria-label="${ifDefined(this.ariaLabel)}"
       class="md3-segmented-button-set ${classMap(this.getRenderClasses())}">
       <slot></slot>
     </span>
     `;
    }
    /** @soyTemplate */
    getRenderClasses() {
        return {};
    }
}
__decorate([
    property({ type: Boolean }),
    __metadata("design:type", Object)
], SegmentedButtonSet.prototype, "multiselect", void 0);
__decorate([
    ariaProperty // tslint:disable-line:no-new-decorators
    ,
    property({ type: String, attribute: 'aria-label' }),
    __metadata("design:type", String)
], SegmentedButtonSet.prototype, "ariaLabel", void 0);
__decorate([
    queryAssignedElements({ flatten: true }),
    __metadata("design:type", Array)
], SegmentedButtonSet.prototype, "buttons", void 0);
//# sourceMappingURL=segmented-button-set.js.map