/**
 * @license
 * Copyright 2023 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
/**
 * Given a surface, an anchor, corners, and some options, this surface will
 * calculate the position of a surface to align the two given corners and keep
 * the surface inside the window viewport. It also provides a StyleInfo map that
 * can be applied to the surface to handle visiblility and position.
 */
export class SurfacePositionController {
    /**
     * @param host The host to connect the controller to.
     * @param getProperties A function that returns the properties for the
     * controller.
     */
    constructor(host, getProperties) {
        this.host = host;
        this.getProperties = getProperties;
        // The current styles to apply to the surface.
        this.surfaceStylesInternal = {
            'display': 'none',
        };
        // Previous values stored for change detection. Open change detection is
        // calculated separately so initialize it here.
        this.lastValues = { isOpen: false };
        this.host.addController(this);
    }
    /**
     * The StyleInfo map to apply to the surface via Lit's stylemap
     */
    get surfaceStyles() {
        return this.surfaceStylesInternal;
    }
    /**
     * Calculates the surface's new position required so that the surface's
     * `surfaceCorner` aligns to the anchor's `anchorCorner` while keeping the
     * surface inside the window viewport. This positioning also respects RTL by
     * checking `getComputedStyle()` on the surface element.
     */
    async position() {
        const { surfaceEl, anchorEl, anchorCorner: anchorCornerRaw, surfaceCorner: surfaceCornerRaw, isTopLayer: topLayerRaw, xOffset, yOffset, } = this.getProperties();
        const anchorCorner = anchorCornerRaw.toUpperCase().trim();
        const surfaceCorner = surfaceCornerRaw.toUpperCase().trim();
        if (!surfaceEl || !anchorEl) {
            return;
        }
        // Paint the surface transparently so that we can get the position and the
        // rect info of the surface.
        this.surfaceStylesInternal = {
            'display': 'block',
            'opacity': '0',
        };
        // Wait for it to be visible.
        this.host.requestUpdate();
        await this.host.updateComplete;
        const surfaceRect = surfaceEl.getBoundingClientRect();
        const anchorRect = anchorEl.getBoundingClientRect();
        const [surfaceBlock, surfaceInline] = surfaceCorner.split('_');
        const [anchorBlock, anchorInline] = anchorCorner.split('_');
        // We use number booleans to multiply values rather than `if` / ternary
        // statements because it _heavily_ cuts down on nesting and readability
        const isTopLayer = topLayerRaw ? 1 : 0;
        // LTR depends on the direction of the SURFACE not the anchor.
        const isLTR = getComputedStyle(surfaceEl).direction === 'ltr' ? 1 : 0;
        const isRTL = isLTR ? 0 : 1;
        const isSurfaceInlineStart = surfaceInline === 'START' ? 1 : 0;
        const isSurfaceInlineEnd = surfaceInline === 'END' ? 1 : 0;
        const isSurfaceBlockStart = surfaceBlock === 'START' ? 1 : 0;
        const isSurfaceBlockEnd = surfaceBlock === 'END' ? 1 : 0;
        const isOneInlineEnd = anchorInline !== surfaceInline ? 1 : 0;
        const isOneBlockEnd = anchorBlock !== surfaceBlock ? 1 : 0;
        /*
         * A diagram that helps describe some of the variables used in the following
         * calculations.
         *
         * ┌───── inline/blockTopLayerOffset
         * │       │
         * │     ┌─▼───┐                  Window
         * │    ┌┼─────┴────────────────────────┐
         * │    ││                              │
         * └──► ││  ┌──inline/blockAnchorOffset │
         *      ││  │     │                     │
         *      └┤  │  ┌──▼───┐                 │
         *       │  │ ┌┼──────┤                 │
         *       │  └─►│Anchor│                 │
         *       │    └┴──────┘                 │
         *       │                              │
         *       │     ┌────────────────────────┼────┐
         *       │     │ Surface                │    │
         *       │     │                        │    │
         *       │     │                        │    │
         *       │     │                        │    │
         *       │     │                        │    │
         *       │     │                        │    │
         *       └─────┼────────────────────────┘    ├┐
         *             │ inline/blockOOBCorrection   ││
         *             │                         │   ││
         *             │                         ├──►││
         *             │                         │   ││
         *             └────────────────────────┐▼───┼┘
         *                                      └────┘
         */
        // Whether or not to apply the width of the anchor
        const inlineAnchorOffset = isOneInlineEnd * anchorRect.width + xOffset;
        // The inline position of the anchor relative to window in LTR
        const inlineTopLayerOffsetLTR = isSurfaceInlineStart * anchorRect.left +
            isSurfaceInlineEnd * (window.innerWidth - anchorRect.right);
        // The inline position of the anchor relative to window in RTL
        const inlineTopLayerOffsetRTL = isSurfaceInlineStart * (window.innerWidth - anchorRect.right) +
            isSurfaceInlineEnd * anchorRect.left;
        // The inline position of the anchor relative to window
        const inlineTopLayerOffset = isLTR * inlineTopLayerOffsetLTR + isRTL * inlineTopLayerOffsetRTL;
        // If the surface's inline would be out of bounds of the window, move it
        // back in
        const inlineOutOfBoundsCorrection = Math.min(0, window.innerWidth - inlineTopLayerOffset - inlineAnchorOffset -
            surfaceRect.width);
        // The inline logical value of the surface
        const inline = isTopLayer * inlineTopLayerOffset + inlineAnchorOffset +
            inlineOutOfBoundsCorrection;
        // Whether or not to apply the height of the anchor
        const blockAnchorOffset = isOneBlockEnd * anchorRect.height + yOffset;
        // The absolute block position of the anchor relative to window
        const blockTopLayerOffset = isSurfaceBlockStart * anchorRect.top +
            isSurfaceBlockEnd * (window.innerHeight - anchorRect.bottom);
        // If the surface's block would be out of bounds of the window, move it back
        // in
        const blockOutOfBoundsCorrection = Math.min(0, window.innerHeight - blockTopLayerOffset - blockAnchorOffset -
            surfaceRect.height);
        // The block logical value of the surface
        const block = isTopLayer * blockTopLayerOffset + blockAnchorOffset +
            blockOutOfBoundsCorrection;
        const surfaceBlockProperty = surfaceBlock === 'START' ? 'inset-block-start' : 'inset-block-end';
        const surfaceInlineProperty = surfaceInline === 'START' ? 'inset-inline-start' : 'inset-inline-end';
        this.surfaceStylesInternal = {
            'display': 'block',
            'opacity': '1',
            [surfaceBlockProperty]: `${block}px`,
            [surfaceInlineProperty]: `${inline}px`,
        };
        this.host.requestUpdate();
    }
    hostUpdate() {
        this.onUpdate();
    }
    hostUpdated() {
        this.onUpdate();
    }
    /**
     * Checks whether the properties passed into the controller have changed since
     * the last positioning. If so, it will reposition if the surface is open or
     * close it if the surface should close.
     */
    async onUpdate() {
        const props = this.getProperties();
        let hasChanged = false;
        for (const [key, value] of Object.entries(props)) {
            // tslint:disable-next-line
            hasChanged = hasChanged || (value !== this.lastValues[key]);
            if (hasChanged)
                break;
        }
        const openChanged = this.lastValues.isOpen !== props.isOpen;
        const hasAnchor = !!props.anchorEl;
        const hasSurface = !!props.surfaceEl;
        if (hasChanged && hasAnchor && hasSurface) {
            // Only update isOpen, because if it's closed, we do not want to waste
            // time on a useless reposition calculation. So save the other "dirty"
            // values until next time it opens.
            this.lastValues.isOpen = props.isOpen;
            if (props.isOpen) {
                // We are going to do a reposition, so save the prop values for future
                // dirty checking.
                this.lastValues = props;
                await this.position();
                props.onOpen();
            }
            else if (openChanged) {
                await props.beforeClose();
                this.close();
                props.onClose();
            }
        }
    }
    /**
     * Hides the surface.
     */
    close() {
        this.surfaceStylesInternal = {
            'display': 'none',
        };
        this.host.requestUpdate();
    }
}
//# sourceMappingURL=surfacePositionController.js.map