/**
 * @license
 * Copyright 2023 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import '../../elevation/elevation.js';
import { LitElement, PropertyValues } from 'lit';
/**
 * Default close action.
 */
export declare const CLOSE_ACTION = "close";
/**
 * A dialog component.
 */
export declare class Dialog extends LitElement {
    private static preventedScrollingCount;
    private static scrollbarTester;
    private static setDocumentScrollingDisabled;
    /**
     * Opens the dialog when set to `true` and closes it when set to `false`.
     */
    open: boolean;
    /**
     * Setting fullscreen displays the dialog fullscreen on small screens.
     * This can be customized via the `fullscreenBreakpoint` property.
     * When showing fullscreen, the header will take up less vertical space, and
     * the dialog will have a `showing-fullscreen`attribute, allowing content to
     * be styled in this state.
     *
     * Dialogs can be sized by setting:
     *
     * * --md-dialog-container-min-block-size
     * * --md-dialog-container-max-block-size
     * * --md-dialog-container-min-inline-size
     * * --md-dialog-container-max-inline-size
     *
     * These are typically configured via media queries and are independent of the
     * fullscreen setting.
     */
    fullscreen: boolean;
    /**
     * A media query string specifying the breakpoint at which the dialog
     * should be shown fullscreen. Note, this only applies when the `fullscreen`
     * property is set.
     *
     * By default, the dialog is shown fullscreen on screens less than 600px wide
     * or 400px tall.
     */
    fullscreenBreakpoint: string;
    /**
     * Hides the dialog footer, making any content slotted into the footer
     * inaccessible.
     */
    footerHidden: boolean;
    /**
     * Renders footer content in a vertically stacked alignment rather than the
     * normal horizontal alignment.
     */
    stacked: boolean;
    /**
     * When the dialog is closed it disptaches `closing` and `closed` events.
     * These events have an action property which has a default value of
     * the value of this property. Specific actions have explicit values but when
     * a value is not specified, the default is used. For example, clicking the
     * scrim, pressing escape, or clicking a button with an action attribute set
     * produce an explicit action.
     *
     * Defaults to `close`.
     */
    defaultAction: string;
    /**
     * The name of an attribute which can be placed on any element slotted into
     * the dialog. If an element has an action attribute set, clicking it will
     * close the dialog and the `closing` and `closed` events dispatched will
     * have their action property set the the value of this attribute on the
     * clicked element.The default valus is `dialogAction`. For example,
     *
     *   <md-dialog>
     *    Content
     *     <md-filled-button slot="footer"dialogAction="buy">
     *       Buy
     *     </md-filled-button>
     *   </md-dialog>
     */
    actionAttribute: string;
    /**
     * When the dialog is opened, it will focus the first element which has
     * an attribute name matching this property. The default value is
     * `dialogFocus`. For example:
     *
     *  <md-dialog>
     *    <md-filled-text-field
     *      label="Enter some text"
     *      dialogFocus
     *    >
     *    </md-filled-text-field>
     *  </md-dialog>
     */
    focusAttribute: string;
    /**
     * Clicking on the scrim surrounding the dialog closes the dialog.
     * The `closing` and `closed` events this produces have an `action` property
     * which is the value of this property and defaults to `close`.
     */
    scrimClickAction: string;
    /**
     * Pressing the `escape` key while the dialog is open closes the dialog.
     * The `closing` and `closed` events this produces have an `action` property
     * which is the value of this property and defaults to `close`.
     */
    escapeKeyAction: string;
    /**
     * When opened, the dialog is displayed modeless or non-modal. This
     * allows users to interact with content outside the dialog without
     * closing the dialog and does not display the scrim around the dialog.
     */
    modeless: boolean;
    /**
     * Set to make the dialog position draggable.
     */
    draggable: boolean;
    private readonly throttle;
    private readonly dialogElement;
    private readonly footerSlot;
    private readonly contentSlot;
    private readonly contentElement;
    private readonly containerElement;
    private readonly headerElement;
    /**
     * Private properties that reflect for styling manually in `updated`.
     */
    private showingFullscreen;
    private showingOpen;
    private opening;
    private closing;
    /**
     * Transition kind. Supported options include: grow, shrink, grow-down,
     * grow-up, grow-left, and grow-right.
     *
     * Defaults to grow-down.
     */
    transition: string;
    private currentAction;
    private dragging;
    private readonly dragMargin;
    private dragInfo?;
    /**
     * Opens and shows the dialog. This is equivalent to setting the `open`
     * property to true.
     */
    show(): void;
    /**
     * Closes the dialog. This is equivalent to setting the `open`
     * property to false.
     */
    close(action?: string): void;
    /**
     * Opens and shows the dialog if it is closed; otherwise closes it.
     */
    toggleShow(): void;
    private getContentScrollInfo;
    protected render(): import("lit-html").TemplateResult<1>;
    protected willUpdate(changed: PropertyValues): void;
    protected firstUpdated(): void;
    protected updated(changed: PropertyValues): void;
    /**
     * Internal state is reflected here as attributes to effect styling. This
     * could be done via internal classes, but it's published on the host
     * to facilitate the (currently undocumented) possibility of customizing
     * styling of user content based on these states.
     * Note, in the future this could be done with `:state(...)` when browser
     * support improves.
     */
    private reflectStateProp;
    private dialogClosedResolver?;
    protected performTransition(shouldDispatchAction: boolean): Promise<void>;
    protected dispatchActionEvent(type: string): void;
    private fullscreenQuery?;
    private fullscreenQueryListener;
    protected updateFullscreen(): void;
    protected handleDialogDismiss(e: Event): void;
    protected handleDialogClick(e: Event): void;
    protected handleContentScroll(): void;
    protected getFocusElement(): HTMLElement | null;
    focus(): void;
    blur(): void;
    protected canStartDrag(e: PointerEvent): boolean;
    protected handlePointerMove(e: PointerEvent): void;
    protected handleDragEnd(e: PointerEvent): void;
}
