// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CONTENT_BROWSER_DEVTOOLS_WEB_CONTENTS_DEVTOOLS_AGENT_HOST_H_
#define CONTENT_BROWSER_DEVTOOLS_WEB_CONTENTS_DEVTOOLS_AGENT_HOST_H_

#include "content/browser/devtools/devtools_agent_host_impl.h"
#include "content/common/content_export.h"
#include "content/public/browser/render_process_host_observer.h"
#include "content/public/browser/web_contents_observer.h"
#include "third_party/abseil-cpp/absl/types/optional.h"

namespace content {

class FrameTreeNode;
class Portal;

class CONTENT_EXPORT WebContentsDevToolsAgentHost
    : public DevToolsAgentHostImpl,
      public WebContentsObserver {
 public:
  // Returns appropriate agent host for given Web Contents
  static WebContentsDevToolsAgentHost* GetFor(WebContents* web_contents);
  // Similar to GetFor(), but creates a host if it doesn't exist yet.
  static WebContentsDevToolsAgentHost* GetOrCreateFor(
      WebContents* web_contents);

  WebContentsDevToolsAgentHost(const WebContentsDevToolsAgentHost&) = delete;
  WebContentsDevToolsAgentHost& operator=(const WebContentsDevToolsAgentHost&) =
      delete;

  static void AddAllAgentHosts(DevToolsAgentHost::List* result);

  // DevToolsAgentHostImpl overrides.
  protocol::TargetAutoAttacher* auto_attacher() override;

  // Instrumentation methods
  void PortalActivated(const Portal& portal);
  void WillInitiatePrerender(FrameTreeNode* ftn);
  // TODO(caseq): do we need more specific signals here?
  void UpdateChildFrameTrees(bool update_target_info);
  void InspectElement(RenderFrameHost* frame_host, int x, int y) override;

 private:
  class AutoAttacher;

  explicit WebContentsDevToolsAgentHost(WebContents* wc);
  ~WebContentsDevToolsAgentHost() override;

  // DevToolsAgentHost overrides.
  void DisconnectWebContents() override;
  void ConnectWebContents(WebContents* web_contents) override;
  BrowserContext* GetBrowserContext() override;
  WebContents* GetWebContents() override;
  std::string GetParentId() override;
  std::string GetOpenerId() override;
  std::string GetOpenerFrameId() override;
  bool CanAccessOpener() override;
  std::string GetType() override;
  std::string GetTitle() override;
  std::string GetDescription() override;
  GURL GetURL() override;
  GURL GetFaviconURL() override;
  bool Activate() override;
  void Reload() override;

  bool Close() override;
  base::TimeTicks GetLastActivityTime() override;

  absl::optional<network::CrossOriginEmbedderPolicy>
  cross_origin_embedder_policy(const std::string& id) override;
  absl::optional<network::CrossOriginOpenerPolicy> cross_origin_opener_policy(
      const std::string& id) override;

  // DevToolsAgentHostImpl overrides.
  DevToolsSession::Mode GetSessionMode() override;
  bool AttachSession(DevToolsSession* session, bool acquire_wake_lock) override;

  // WebContentsObserver overrides.
  void WebContentsDestroyed() override;

  DevToolsAgentHostImpl* GetPrimaryFrameAgent();
  scoped_refptr<DevToolsAgentHost> GetOrCreatePrimaryFrameAgent();

  std::unique_ptr<AutoAttacher> auto_attacher_;
};

}  // namespace content

#endif  // CONTENT_BROWSER_DEVTOOLS_WEB_CONTENTS_DEVTOOLS_AGENT_HOST_H_
