// SPDX-FileCopyrightText: 2024 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include "modelinfo.h"

#include <QFileInfo>
#include <QDir>
#include <QJsonDocument>
#include <QJsonObject>
#include <QVariantHash>

GLOBAL_USE_NAMESPACE

ModelInfo::ModelInfo(const QString &path): rootPath(path)
{

}

QString ModelInfo::name() const
{
    QFileInfo info(rootPath);
    return info.fileName();
}

QString ModelInfo::path() const
{
    return rootPath;
}

QStringList ModelInfo::formats() const
{
    QStringList ret;

    QDir dir(rootPath);
    QFileInfoList list = dir.entryInfoList(QDir::NoDotAndDotDot | QDir::Dirs);
    for (const QFileInfo &info : list) {
        if (QFileInfo::exists(info.absoluteFilePath() + "/config.json"))
            ret.append(info.fileName());
    }

    return ret;
}

QStringList ModelInfo::architectures(const QString &format) const
{
    QStringList ret;
    if (format.isEmpty())
        return ret;

    QFile f(imagePath(format) + "/config.json");
    if (f.open(QFile::ReadOnly)) {
        auto doc = QJsonDocument::fromJson(f.readAll());
        f.close();

        auto configs = doc.object().toVariantHash();
        ret = configs.value("architectures").toStringList();
    }

    return ret;
}

QString ModelInfo::imagePath(const QString &format) const
{
    return QString("%0/%1").arg(rootPath).arg(format.toLower());
}

QString ModelInfo::chatTemplate(const QString &format) const
{
    QString ret;
    if (format.isEmpty())
        return ret;

    QFile f(imagePath(format) + "/config.json");
    if (f.open(QFile::ReadOnly)) {
        auto doc = QJsonDocument::fromJson(f.readAll());
        f.close();

        auto configs = doc.object().toVariantHash();
        QString temp = configs.value("template").toString();

        QFile ft(imagePath(format) + "/" + temp);
        if (ft.open(QFile::ReadOnly)) {
            ret = ft.readAll();
            ft.close();
        }
    }

    return ret;
}
