/*
   ub_beta2greek.h - tables to convert from Beta Code
                     to Unicode in Greek.

   Author: Paul Hardy, unifoundry <at> unifoundry.com

   Copyright (C) 2018, 2019, 2020 Paul Hardy

   LICENSE:

      This program is free software: you can redistribute it and/or modify
      it under the terms of the GNU General Public License as published by
      the Free Software Foundation, either version 2 of the License, or
      (at your option) any later version.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.
      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


/*
   The combining marks in polytonic Greek, with classical names.
   These are ORed together to form the row indices in the array
   that follows.
*/
#define UB_GREEK_PSILI          0x40  /* smooth breathing  */
#define UB_GREEK_DASIA          0x20  /* rough breathing   */
#define UB_GREEK_VARIA          0x10  /* grave accent      */
#define UB_GREEK_OXIA           0x08  /* acute accent      */
#define UB_GREEK_PERISPOMENI    0x04  /* circumflex accent */
#define UB_GREEK_DIALYTIKA      0x02  /* diaresis          */
#define UB_GREEK_YPOGEGRAMMENI  0x01  /* iota subscript    */
#define UB_GREEK_PROSGEGRAMMENI 0x01  /* iota subscript    */


/*
   Open and close quotation mark definitions.
*/
#define UB_GREEK_OPEN_DOUBLE_QUOTE	0X00AB
#define UB_GREEK_CLOSE_DOUBLE_QUOTE	0X00BB

#define UB_COPTIC_OPEN_DOUBLE_QUOTE	0X00AB
#define UB_COPTIC_CLOSE_DOUBLE_QUOTE	0X00BB

#define UB_HEBREW_OPEN_DOUBLE_QUOTE	0X201E
#define UB_HEBREW_CLOSE_DOUBLE_QUOTE	0X201D


/*
   Table to convert an ASCII letter into a Unicode Greek letter.

   Middle Sigma, Final Sigma, and Lunate Sigma are handled specially elsewhere.
*/
unsigned ascii2greek[128] = {
/*   0/8    1/9    2/A    3/B    4/C    5/D    6/E    7/F   */
       0,     0,     0,     0,     0,     0,     0,     0,  /* 0x00..0x07 */
       0,  '\t',  '\n',     0,     0,     0,  '\r',     0,  /* 0x08..0x0F */
       0,     0,     0,     0,     0,     0,     0,     0,  /* 0x10..0x17 */
       0,     0,     0,     0,     0,     0,     0,     0,  /* 0x18..0x1F */
     ' ',   '!',   '"', 0x374,   '$',   '%',   '&', 0x2BC,  /* 0x20..0x27  !"#$%&' */
   0x314, 0x313,   '*', 0x308,   ',',0x2010,   '.', 0x301,  /* 0x28..0x2F ()*+,-./ */
     '0',   '1',   '2',   '3',   '4',   '5',   '6',   '7',  /* 0x30..0x37 01234567 */
     '8',   '9',  0xB7,   ';',0x2039, 0x342,0x203A, 0x323,  /* 0x38..0x3F 89:;<=>? */
     '@', 0x391, 0x392, 0x39E, 0x394, 0x395, 0x3A6, 0x393,  /* 0x40..0x47 @ABCDEFG */
   0x397, 0x399, 0x3A2, 0x39A, 0x39B, 0x39C, 0x39D, 0x39F,  /* 0x48..0x4F HIJKLMNO */
   0x3A0, 0x398, 0x3A1, 0x3A3, 0x3A4, 0x3A5, 0x3DC, 0x3A9,  /* 0x50..0x57 PQRSTUVW */
   0x3A7, 0x3A8, 0x396,   '[', 0x300,   ']',   '^',0x2014,  /* 0x58..0x5F XYZ[\]^_ */
     '`', 0x3B1, 0x3B2, 0x3BE, 0x3B4, 0x3B5, 0x3C6, 0x3B3,  /* 0x60..0x67 `abcdefg */
   0x3B7, 0x3B9, 0x3C2, 0x3BA, 0x3BB, 0x3BC, 0x3BD, 0x3BF,  /* 0x68..0x6F hijklmno */
   0x3C0, 0x3B8, 0x3C1, 0x3C3, 0x3C4, 0x3C5, 0x3DD, 0x3C9,  /* 0x70..0x77 pqrstuvw */
   0x3C7, 0x3C8, 0x3B6,   '{', 0x345,   '}',   '~',     0   /* 0x78..0x7F xyz{|}~<DEL> */
/*   0/8    1/9    2/A    3/B    4/C    5/D    6/E    7/F   */
};


/*
   Table of combining marks for Greek.

               0 --> not a combining mark
        non-zero --> Unicode code point of Beta Code combining mark
*/
unsigned greek_comb2uni[128] = {
/*   0/8    1/9    2/A    3/B    4/C    5/D    6/E    7/F   */
       0,     0,     0,     0,     0,     0,     0,     0,   /* 0x00..0x07 */
       0,     0,     0,     0,     0,     0,     0,     0,   /* 0x08..0x0F */
       0,     0,     0,     0,     0,     0,     0,     0,   /* 0x10..0x17 */
       0,     0,     0,     0,     0,     0,     0,     0,   /* 0x18..0x1F */
       0,     0,     0,     0,     0,     0,     0,     0,   /* 0x20..0x27 " !"#$%&'" */
   0x314, 0x313,     0, 0x308,     0,     0,     0, 0x301,   /* 0x28..0x2F "()*+,-./" */
       0,     0,     0,     0,     0,     0,     0,     0,   /* 0x30..0x37 "01234567" */
       0,     0,     0,     0,     0, 0x342,     0,     0,   /* 0x38..0x3F "89:;<=>?" */
       0,     0,     0,     0,     0,     0,     0,     0,   /* 0x40..0x47 "@ABCDEFG" */
       0,     0,     0,     0,     0,     0,     0,     0,   /* 0x48..0x4F "HIJKLMNO" */
       0,     0,     0,     0,     0,     0,     0,     0,   /* 0x50..0x57 "PQRSTUVW" */
       0,     0,     0,     0, 0x300,     0,     0,     0,   /* 0x58..0x5F "XYZ[\]^_" */
       0,     0,     0,     0,     0,     0,     0,     0,   /* 0x60..0x67 "`abcdefg" */
       0,     0,     0,     0,     0,     0,     0,     0,   /* 0x68..0x6F "hijklmno" */
       0,     0,     0,     0,     0,     0,     0,     0,   /* 0x70..0x77 "pqrstuvw" */
       0,     0,     0,     0, 0x345,     0,     0,     0,   /* 0x78..0x7F "xyz{|}~" and <DEL> */
/*   0/8    1/9    2/A    3/B    4/C    5/D    6/E    7/F   */
};


/*
   beta2combining - array of pre-formed Unicode polytonic
                    Greek code points, ordered by the types
                    of combining marks in a binary progression.

   The rows are indexed as follows:

         Row
      Index Bit  Combining Mark
      ---------  --------------
         0x80    Unused
         0x40    Psili
         0x20    Dasia
         0x10    Varia
         0x08    Oxia
         0x04    Perispomeni
         0x02    Dialytika
         0x01    Ypogegammini

   Note that these sets of marks are mutually exclusive:

        Psili and Dasia
        Varia, Oxia, and Perispomeni

   They all were given separate indices to simplify
   filling in the grid of code points.

   The letters that those combining marks affect are
   the column (second dimension) values:

        Upper-case (Beta Code) A, E, H, I, O, R, U, W, and then
        Lower-case (Beta Code) a, e, h, i, o, r, u,  and w.

   This internal coding is converted into these
   corresponding Greek letters:

        alpha, epsilon, eta, iota, omicron, rho, upsilon, omega.

   Beta Code itself is case-insensitive; upper-case Beta Code
   letters are designated with a preceding '*' character.
   The unibetacode library converts received Beta Code into
   upper-case Latin letters for upper-case Greek, and into
   lower-case Latin letters for lower-case Greek.

   Code points are taken from these Unicode blocks:

        U+0370..U+03FF Greek and Coptic
        U+1F00..U+1FFF Greek Extended

    A value of 0 in the array designates that there is no
    mapping to a pre-formed Unicode code point.  In that
    case, software must construct an output string of the
    letter followed by one or more combining characters.
*/
unsigned beta2combining [128][16] = {
/*
                         P
                        Yr
                        po
                    P   os
                    e   gg
                    r D ee
                    i i gg
                    s a rr
                    p l aa
            P D V   o y mm
            s a a O m t mm
            i s r x e i ee
            l i i i n k nn      A       E       H       I       O       R       U       W
   Dec  Hex i a a a i a ii      a       e       h       i       o       r       u       w    */

/*   0 0x00 0,0,0,0,0,0,0 */ 0x0391, 0x0395, 0x0397, 0x0399, 0x039F, 0x03A1, 0x03A5, 0x03A9, /* A E H I O R U W */
                             0x03B1, 0x03B5, 0x03B7, 0x03B9, 0x03BF, 0x03C1, 0x03C5, 0x03C9,
/*   1 0x01 0,0,0,0,0,0,1 */ 0x1FBC,      0, 0x1FCC,      0,      0,      0,      0, 0x1FFC, /* A E H I O R U W */
                             0x1FB3,      0, 0x1FC3,      0,      0,      0,      0, 0x1FF3,
/*   2 0x02 0,0,0,0,0,1,0 */      0,      0,      0, 0x03AA,      0,      0, 0x03AB,      0, /* A E H I O R U W */
                                  0,      0,      0, 0x03CA,      0,      0, 0x03CB,      0,
/*   3 0x03 0,0,0,0,0,1,1 */      0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
                                  0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
/*   4 0x04 0,0,0,0,1,0,0 */      0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
                             0x1FB6,      0, 0x1FC6, 0x1FD6,      0,      0, 0x1FE6, 0x1FF6,
/*   5 0x05 0,0,0,0,1,0,1 */      0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
                             0x1FB7,      0, 0x1FC7,      0,      0,      0,      0, 0x1FF7,
/*   6 0x06 0,0,0,0,1,1,0 */      0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
                                  0,      0,      0, 0x1FD7,      0,      0, 0x1FE7,      0,
/*   7 0x07 0,0,0,0,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*   8 0x08 0,0,0,1,0,0,0 */ 0x0386, 0x0388, 0x0389, 0x038A, 0x038C,      0, 0x038E, 0x038F, /* A E H I O R U W */
                             0x03AC, 0x03AD, 0x03AE, 0x03AF, 0x03CC,      0, 0x03CD, 0x03CE,
/*   9 0x09 0,0,0,1,0,0,1 */      0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
                             0x1FB4,      0, 0x1FC4,      0,      0,      0,      0, 0x1FF4,
/*  10 0x0A 0,0,0,1,0,1,0 */      0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
                                  0,      0,      0, 0x0390,      0,      0, 0x03B0,      0,
/*  11 0x0B 0,0,0,1,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  12 0x0C 0,0,0,1,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  13 0x0D 0,0,0,1,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  14 0x0E 0,0,0,1,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  15 0x0F 0,0,0,1,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  16 0x10 0,0,1,0,0,0,0 */ 0x1FBA, 0x1FC8, 0x1FCA, 0x1FDA, 0x1FF8,      0, 0x1FEA, 0x1FFA, /* A E H I O R U W */
                             0x1F70, 0x1F72, 0x1F74, 0x1F76, 0x1F78,      0, 0x1F7A, 0x1F7C,
/*  17 0x11 0,0,1,0,0,0,1 */      0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
                             0x1FB2,      0, 0x1FC2,      0,      0,      0,      0, 0x1FF2,
/*  18 0x12 0,0,1,0,0,1,0 */      0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
                                  0,      0,      0, 0x1FD2,      0,      0, 0x1FE2,      0,
/*  19 0x13 0,0,1,0,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  20 0x14 0,0,1,0,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  21 0x15 0,0,1,0,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  22 0x16 0,0,1,0,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  23 0x17 0,0,1,0,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  24 0x18 0,0,1,1,0,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  25 0x19 0,0,1,1,0,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  26 0x1A 0,0,1,1,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  27 0x1B 0,0,1,1,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  28 0x1C 0,0,1,1,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  29 0x1D 0,0,1,1,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  30 0x1E 0,0,1,1,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  31 0x1F 0,0,1,1,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  32 0x20 0,1,0,0,0,0,0 */ 0X1F09, 0X1F19, 0X1F29, 0X1F39, 0X1F49, 0x1FEC, 0X1F59, 0X1F69, /* A E H I O R U W */
                             0x1F01, 0x1F11, 0x1F21, 0x1F31, 0x1F41, 0x1FE5, 0x1F51, 0x1F61,
/*  33 0x21 0,1,0,0,0,0,1 */ 0x1F89,      0, 0x1F99,      0,      0,      0,      0, 0x1FA9, /* A E H I O R U W */
                             0x1F81,      0, 0x1F91,      0,      0,      0,      0, 0x1FA1,
/*  34 0x22 0,1,0,0,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  35 0x23 0,1,0,0,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  36 0x24 0,1,0,0,1,0,0 */ 0x1F0F,      0, 0x1F2F, 0x1F3F,      0,      0, 0x1F5F, 0x1F6F, /* A E H I O R U W */
                             0X1F07,      0, 0X1F27, 0X1F37,      0,      0, 0X1F57, 0X1F67,
/*  37 0x25 0,1,0,0,1,0,1 */ 0x1F8F,      0, 0x1F9F,      0,      0,      0,      0, 0x1FAF, /* A E H I O R U W */
                             0x1F87,      0, 0x1F97,      0,      0,      0,      0, 0x1FA7,
/*  38 0x26 0,1,0,0,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  39 0x27 0,1,0,0,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  40 0x28 0,1,0,1,0,0,0 */ 0x1F0D, 0x1F1D, 0x1F2D, 0x1F3D, 0x1F4D,      0, 0x1F5D, 0x1F6D, /* A E H I O R U W */
                             0x1F05, 0x1F15, 0x1F25, 0x1F35, 0x1F45,      0, 0x1F55, 0x1F65,
/*  41 0x29 0,1,0,1,0,0,1 */ 0x1F8D,      0, 0x1F9D,      0,      0,      0,      0, 0x1FAD, /* A E H I O R U W */
                             0x1F85,      0, 0x1F95,      0,      0,      0,      0, 0x1FA5,
/*  42 0x2A 0,1,0,1,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  43 0x2B 0,1,0,1,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  44 0x2C 0,1,0,1,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  45 0x2D 0,1,0,1,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  46 0x2E 0,1,0,1,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  47 0x2F 0,1,0,1,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  48 0x30 0,1,1,0,0,0,0 */ 0x1F0B, 0x1F1B, 0x1F2B, 0x1F3B, 0x1F4B,      0, 0x1F5B, 0x1F6B, /* A E H I O R U W */
                             0x1F03, 0x1F13, 0x1F23, 0x1F33, 0x1F43,      0, 0x1F53, 0x1F63,
/*  49 0x31 0,1,1,0,0,0,1 */ 0x1F8B,      0, 0x1F9B,      0,      0,      0,      0, 0x1FAB, /* A E H I O R U W */
                             0x1F83,      0, 0x1F93,      0,      0,      0,      0, 0x1FA3,
/*  50 0x32 0,1,1,0,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  51 0x33 0,1,1,0,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  52 0x34 0,1,1,0,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  53 0x35 0,1,1,0,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  54 0x36 0,1,1,0,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  55 0x37 0,1,1,0,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  56 0x38 0,1,1,1,0,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  57 0x39 0,1,1,1,0,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  58 0x3A 0,1,1,1,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  59 0x3B 0,1,1,1,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  60 0x3C 0,1,1,1,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  61 0x3D 0,1,1,1,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  62 0x3E 0,1,1,1,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  63 0x3F 0,1,1,1,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  64 0x40 1,0,0,0,0,0,0 */ 0X1F08, 0X1F18, 0X1F28, 0X1F38, 0X1F48,      0,      0, 0X1F68, /* A E H I O R U W */
                             0x1F00, 0x1F10, 0x1F20, 0x1F30, 0x1F40, 0x1FE4, 0x1F50, 0x1F60,
/*  65 0x41 1,0,0,0,0,0,1 */ 0x1F88,      0, 0x1F98,      0,      0,      0,      0, 0x1FA8, /* A E H I O R U W */
                             0x1F80,      0, 0x1F90,      0,      0,      0,      0, 0x1FA0,
/*  66 0x42 1,0,0,0,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  67 0x43 1,0,0,0,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  68 0x44 1,0,0,0,1,0,0 */ 0x1F0E,      0, 0x1F2E, 0x1F3E,      0,      0,      0, 0x1F6E, /* A E H I O R U W */
                             0X1F06,      0, 0X1F26, 0X1F36,      0,      0, 0X1F56, 0X1F66,
/*  69 0x45 1,0,0,0,1,0,1 */ 0x1F8E,      0, 0x1F9E,      0,      0,      0,      0, 0x1FAE, /* A E H I O R U W */
                             0x1F86,      0, 0x1F96,      0,      0,      0,      0, 0x1FA6,
/*  70 0x46 1,0,0,0,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  71 0x47 1,0,0,0,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  72 0x48 1,0,0,1,0,0,0 */ 0x1F0C, 0x1F1C, 0x1F2C, 0x1F3C, 0x1F4C,      0,      0, 0x1F6C, /* A E H I O R U W */
                             0x1F04, 0x1F14, 0x1F24, 0x1F34, 0x1F44,      0, 0x1F54, 0x1F64,
/*  73 0x49 1,0,0,1,0,0,1 */ 0x1F8C,      0, 0x1F9C,      0,      0,      0,      0, 0x1FAC, /* A E H I O R U W */
                             0x1F84,      0, 0x1F94,      0,      0,      0,      0, 0x1FA4,
/*  74 0x4A 1,0,0,1,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  75 0x4B 1,0,0,1,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  76 0x4C 1,0,0,1,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  77 0x4D 1,0,0,1,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  78 0x4E 1,0,0,1,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  79 0x4F 1,0,0,1,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  80 0x50 1,0,1,0,0,0,0 */ 0x1F0A, 0x1F1A, 0x1F2A, 0x1F3A, 0x1F4A,      0,      0, 0x1F6A, /* A E H I O R U W */
                             0x1F02, 0x1F12, 0x1F22, 0x1F32, 0x1F42,      0, 0x1F52, 0x1F62,
/*  81 0x51 1,0,1,0,0,0,1 */ 0x1F8A,      0, 0x1F9A,      0,      0,      0,      0, 0x1FAA, /* A E H I O R U W */
                             0x1F82,      0, 0x1F92,      0,      0,      0,      0, 0x1FA2,
/*  82 0x52 1,0,1,0,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  83 0x53 1,0,1,0,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  84 0x54 1,0,1,0,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  85 0x55 1,0,1,0,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  86 0x56 1,0,1,0,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  87 0x57 1,0,1,0,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  88 0x58 1,0,1,1,0,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  89 0x59 1,0,1,1,0,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  90 0x5A 1,0,1,1,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  91 0x5B 1,0,1,1,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  92 0x5C 1,0,1,1,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  93 0x5D 1,0,1,1,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  94 0x5E 1,0,1,1,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  95 0x5F 1,0,1,1,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  96 0x60 1,1,0,0,0,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  97 0x61 1,1,0,0,0,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  98 0x62 1,1,0,0,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  99 0x63 1,1,0,0,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 100 0x64 1,1,0,0,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 101 0x65 1,1,0,0,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 102 0x66 1,1,0,0,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 103 0x67 1,1,0,0,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 104 0x68 1,1,0,1,0,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 105 0x69 1,1,0,1,0,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 106 0x6A 1,1,0,1,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 107 0x6B 1,1,0,1,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 108 0x6C 1,1,0,1,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 109 0x6D 1,1,0,1,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 110 0x6E 1,1,0,1,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 111 0x6F 1,1,0,1,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 112 0x70 1,1,1,0,0,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 113 0x71 1,1,1,0,0,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 114 0x72 1,1,1,0,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 115 0x73 1,1,1,0,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 116 0x74 1,1,1,0,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 117 0x75 1,1,1,0,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 118 0x76 1,1,1,0,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 119 0x77 1,1,1,0,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 120 0x78 1,1,1,1,0,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 121 0x79 1,1,1,1,0,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 122 0x7A 1,1,1,1,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 123 0x7B 1,1,1,1,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 124 0x7C 1,1,1,1,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 125 0x7D 1,1,1,1,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 126 0x7E 1,1,1,1,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 127 0x7F 1,1,1,1,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0
};


/*
   This is the same as the previous array, except no polytonic glyphs
   are taken from the Greek and Coptic block, U+0370..U+03FF.  All
   polytonic glyphs are taken from the Greek Extended block,
   U+1F00..U+1FFF.  This can be desirable for use with fonts that
   draw letter glyphs significantly differently between the two
   blocks.
*/
unsigned beta2combining_alt [128][16] = {
/*
                         P
                        Yr
                        po
                    P   os
                    e   gg
                    r D ee
                    i i gg
                    s a rr
                    p l aa
            P D V   o y mm
            s a a O m t mm
            i s r x e i ee
            l i i i n k nn      A       E       H       I       O       R       U       W
   Dec  Hex i a a a i a ii      a       e       h       i       o       r       u       w    */

/*   0 0x00 0,0,0,0,0,0,0 */ 0x0391, 0x0395, 0x0397, 0x0399, 0x039F, 0x03A1, 0x03A5, 0x03A9, /* A E H I O R U W */
                             0x03B1, 0x03B5, 0x03B7, 0x03B9, 0x03BF, 0x03C1, 0x03C5, 0x03C9,
/*   1 0x01 0,0,0,0,0,0,1 */ 0x1FBC,      0, 0x1FCC,      0,      0,      0,      0, 0x1FFC, /* A E H I O R U W */
                             0x1FB3,      0, 0x1FC3,      0,      0,      0,      0, 0x1FF3,
/*   2 0x02 0,0,0,0,0,1,0 */      0,      0,      0, 0x03AA,      0,      0, 0x03AB,      0, /* A E H I O R U W */
                                  0,      0,      0, 0x03CA,      0,      0, 0x03CB,      0,
/*   3 0x03 0,0,0,0,0,1,1 */      0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
                                  0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
/*   4 0x04 0,0,0,0,1,0,0 */      0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
                             0x1FB6,      0, 0x1FC6, 0x1FD6,      0,      0, 0x1FE6, 0x1FF6,
/*   5 0x05 0,0,0,0,1,0,1 */      0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
                             0x1FB7,      0, 0x1FC7,      0,      0,      0,      0, 0x1FF7,
/*   6 0x06 0,0,0,0,1,1,0 */      0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
                                  0,      0,      0, 0x1FD7,      0,      0, 0x1FE7,      0,
/*   7 0x07 0,0,0,0,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*   8 0x08 0,0,0,1,0,0,0 */ 0x1FBB, 0x1FC9, 0x1FCB, 0x1FDB, 0x1FF9,      0, 0x1FEB, 0x1FFB, /* A E H I O R U W */
                             0x1F71, 0x1F73, 0x1F75, 0x1F77, 0x1F79,      0, 0x1F7B, 0x1F7D,
/*   9 0x09 0,0,0,1,0,0,1 */      0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
                             0x1FB4,      0, 0x1FC4,      0,      0,      0,      0, 0x1FF4,
/*  10 0x0A 0,0,0,1,0,1,0 */      0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
                                  0,      0,      0, 0x1FD3,      0,      0, 0x1FE3,      0,
/*  11 0x0B 0,0,0,1,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  12 0x0C 0,0,0,1,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  13 0x0D 0,0,0,1,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  14 0x0E 0,0,0,1,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  15 0x0F 0,0,0,1,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  16 0x10 0,0,1,0,0,0,0 */ 0x1FBA, 0x1FC8, 0x1FCA, 0x1FDA, 0x1FF8,      0, 0x1FEA, 0x1FFA, /* A E H I O R U W */
                             0x1F70, 0x1F72, 0x1F74, 0x1F76, 0x1F78,      0, 0x1F7A, 0x1F7C,
/*  17 0x11 0,0,1,0,0,0,1 */      0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
                             0x1FB2,      0, 0x1FC2,      0,      0,      0,      0, 0x1FF2,
/*  18 0x12 0,0,1,0,0,1,0 */      0,      0,      0,      0,      0,      0,      0,      0, /* A E H I O R U W */
                                  0,      0,      0, 0x1FD2,      0,      0, 0x1FE2,      0,
/*  19 0x13 0,0,1,0,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  20 0x14 0,0,1,0,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  21 0x15 0,0,1,0,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  22 0x16 0,0,1,0,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  23 0x17 0,0,1,0,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  24 0x18 0,0,1,1,0,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  25 0x19 0,0,1,1,0,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  26 0x1A 0,0,1,1,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  27 0x1B 0,0,1,1,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  28 0x1C 0,0,1,1,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  29 0x1D 0,0,1,1,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  30 0x1E 0,0,1,1,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  31 0x1F 0,0,1,1,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  32 0x20 0,1,0,0,0,0,0 */ 0X1F09, 0X1F19, 0X1F29, 0X1F39, 0X1F49, 0x1FEC, 0X1F59, 0X1F69, /* A E H I O R U W */
                             0x1F01, 0x1F11, 0x1F21, 0x1F31, 0x1F41, 0x1FE5, 0x1F51, 0x1F61,
/*  33 0x21 0,1,0,0,0,0,1 */ 0x1F89,      0, 0x1F99,      0,      0,      0,      0, 0x1FA9, /* A E H I O R U W */
                             0x1F81,      0, 0x1F91,      0,      0,      0,      0, 0x1FA1,
/*  34 0x22 0,1,0,0,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  35 0x23 0,1,0,0,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  36 0x24 0,1,0,0,1,0,0 */ 0x1F0F,      0, 0x1F2F, 0x1F3F,      0,      0, 0x1F5F, 0x1F6F, /* A E H I O R U W */
                             0X1F07,      0, 0X1F27, 0X1F37,      0,      0, 0X1F57, 0X1F67,
/*  37 0x25 0,1,0,0,1,0,1 */ 0x1F8F,      0, 0x1F9F,      0,      0,      0,      0, 0x1FAF, /* A E H I O R U W */
                             0x1F87,      0, 0x1F97,      0,      0,      0,      0, 0x1FA7,
/*  38 0x26 0,1,0,0,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  39 0x27 0,1,0,0,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  40 0x28 0,1,0,1,0,0,0 */ 0x1F0D, 0x1F1D, 0x1F2D, 0x1F3D, 0x1F4D,      0, 0x1F5D, 0x1F6D, /* A E H I O R U W */
                             0x1F05, 0x1F15, 0x1F25, 0x1F35, 0x1F45,      0, 0x1F55, 0x1F65,
/*  41 0x29 0,1,0,1,0,0,1 */ 0x1F8D,      0, 0x1F9D,      0,      0,      0,      0, 0x1FAD, /* A E H I O R U W */
                             0x1F85,      0, 0x1F95,      0,      0,      0,      0, 0x1FA5,
/*  42 0x2A 0,1,0,1,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  43 0x2B 0,1,0,1,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  44 0x2C 0,1,0,1,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  45 0x2D 0,1,0,1,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  46 0x2E 0,1,0,1,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  47 0x2F 0,1,0,1,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  48 0x30 0,1,1,0,0,0,0 */ 0x1F0B, 0x1F1B, 0x1F2B, 0x1F3B, 0x1F4B,      0, 0x1F5B, 0x1F6B, /* A E H I O R U W */
                             0x1F03, 0x1F13, 0x1F23, 0x1F33, 0x1F43,      0, 0x1F53, 0x1F63,
/*  49 0x31 0,1,1,0,0,0,1 */ 0x1F8B,      0, 0x1F9B,      0,      0,      0,      0, 0x1FAB, /* A E H I O R U W */
                             0x1F83,      0, 0x1F93,      0,      0,      0,      0, 0x1FA3,
/*  50 0x32 0,1,1,0,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  51 0x33 0,1,1,0,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  52 0x34 0,1,1,0,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  53 0x35 0,1,1,0,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  54 0x36 0,1,1,0,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  55 0x37 0,1,1,0,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  56 0x38 0,1,1,1,0,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  57 0x39 0,1,1,1,0,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  58 0x3A 0,1,1,1,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  59 0x3B 0,1,1,1,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  60 0x3C 0,1,1,1,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  61 0x3D 0,1,1,1,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  62 0x3E 0,1,1,1,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  63 0x3F 0,1,1,1,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  64 0x40 1,0,0,0,0,0,0 */ 0X1F08, 0X1F18, 0X1F28, 0X1F38, 0X1F48,      0,      0, 0X1F68, /* A E H I O R U W */
                             0x1F00, 0x1F10, 0x1F20, 0x1F30, 0x1F40, 0x1FE4, 0x1F50, 0x1F60,
/*  65 0x41 1,0,0,0,0,0,1 */ 0x1F88,      0, 0x1F98,      0,      0,      0,      0, 0x1FA8, /* A E H I O R U W */
                             0x1F80,      0, 0x1F90,      0,      0,      0,      0, 0x1FA0,
/*  66 0x42 1,0,0,0,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  67 0x43 1,0,0,0,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  68 0x44 1,0,0,0,1,0,0 */ 0x1F0E,      0, 0x1F2E, 0x1F3E,      0,      0,      0, 0x1F6E, /* A E H I O R U W */
                             0X1F06,      0, 0X1F26, 0X1F36,      0,      0, 0X1F56, 0X1F66,
/*  69 0x45 1,0,0,0,1,0,1 */ 0x1F8E,      0, 0x1F9E,      0,      0,      0,      0, 0x1FAE, /* A E H I O R U W */
                             0x1F86,      0, 0x1F96,      0,      0,      0,      0, 0x1FA6,
/*  70 0x46 1,0,0,0,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  71 0x47 1,0,0,0,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  72 0x48 1,0,0,1,0,0,0 */ 0x1F0C, 0x1F1C, 0x1F2C, 0x1F3C, 0x1F4C,      0,      0, 0x1F6C, /* A E H I O R U W */
                             0x1F04, 0x1F14, 0x1F24, 0x1F34, 0x1F44,      0, 0x1F54, 0x1F64,
/*  73 0x49 1,0,0,1,0,0,1 */ 0x1F8C,      0, 0x1F9C,      0,      0,      0,      0, 0x1FAC, /* A E H I O R U W */
                             0x1F84,      0, 0x1F94,      0,      0,      0,      0, 0x1FA4,
/*  74 0x4A 1,0,0,1,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  75 0x4B 1,0,0,1,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  76 0x4C 1,0,0,1,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  77 0x4D 1,0,0,1,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  78 0x4E 1,0,0,1,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  79 0x4F 1,0,0,1,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  80 0x50 1,0,1,0,0,0,0 */ 0x1F0A, 0x1F1A, 0x1F2A, 0x1F3A, 0x1F4A,      0,      0, 0x1F6A, /* A E H I O R U W */
                             0x1F02, 0x1F12, 0x1F22, 0x1F32, 0x1F42,      0, 0x1F52, 0x1F62,
/*  81 0x51 1,0,1,0,0,0,1 */ 0x1F8A,      0, 0x1F9A,      0,      0,      0,      0, 0x1FAA, /* A E H I O R U W */
                             0x1F82,      0, 0x1F92,      0,      0,      0,      0, 0x1FA2,
/*  82 0x52 1,0,1,0,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  83 0x53 1,0,1,0,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  84 0x54 1,0,1,0,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  85 0x55 1,0,1,0,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  86 0x56 1,0,1,0,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  87 0x57 1,0,1,0,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  88 0x58 1,0,1,1,0,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  89 0x59 1,0,1,1,0,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  90 0x5A 1,0,1,1,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  91 0x5B 1,0,1,1,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  92 0x5C 1,0,1,1,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  93 0x5D 1,0,1,1,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  94 0x5E 1,0,1,1,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  95 0x5F 1,0,1,1,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  96 0x60 1,1,0,0,0,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  97 0x61 1,1,0,0,0,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  98 0x62 1,1,0,0,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/*  99 0x63 1,1,0,0,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 100 0x64 1,1,0,0,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 101 0x65 1,1,0,0,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 102 0x66 1,1,0,0,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 103 0x67 1,1,0,0,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 104 0x68 1,1,0,1,0,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 105 0x69 1,1,0,1,0,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 106 0x6A 1,1,0,1,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 107 0x6B 1,1,0,1,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 108 0x6C 1,1,0,1,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 109 0x6D 1,1,0,1,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 110 0x6E 1,1,0,1,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 111 0x6F 1,1,0,1,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 112 0x70 1,1,1,0,0,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 113 0x71 1,1,1,0,0,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 114 0x72 1,1,1,0,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 115 0x73 1,1,1,0,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 116 0x74 1,1,1,0,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 117 0x75 1,1,1,0,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 118 0x76 1,1,1,0,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 119 0x77 1,1,1,0,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 120 0x78 1,1,1,1,0,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 121 0x79 1,1,1,1,0,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 122 0x7A 1,1,1,1,0,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 123 0x7B 1,1,1,1,0,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 124 0x7C 1,1,1,1,1,0,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 125 0x7D 1,1,1,1,1,0,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 126 0x7E 1,1,1,1,1,1,0 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
/* 127 0x7F 1,1,1,1,1,1,1 */  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0
};

