// Copyright 2024 the V8 project authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include <map>

#include "src/compiler/backend/instruction.h"
#include "src/compiler/graph-visualizer.h"
#include "src/compiler/turboshaft/assembler.h"
#include "src/compiler/turboshaft/phase.h"
#include "test/unittests/test-utils.h"

namespace v8::internal::compiler::turboshaft {

class TestInstance {
 public:
  using Assembler = TSAssembler<>;

  struct CapturedOperation {
    TestInstance* instance;
    OpIndex input;
    std::set<OpIndex> generated_output;

    template <typename T>
    bool Contains() const {
      for (OpIndex o : generated_output) {
        if (instance->graph().Get(o).Is<T>()) return true;
      }
      return false;
    }
  };

  template <typename Builder>
  static TestInstance CreateFromGraph(int parameter_count,
                                      const Builder& builder, Isolate* isolate,
                                      Zone* zone) {
    auto graph = std::make_unique<Graph>(zone);
    TestInstance instance(std::move(graph), isolate, zone);
    // Generate a function prolog
    Block* start_block = instance.Asm().NewBlock();
    instance.Asm().Bind(start_block);
    instance.Asm().Parameter(3, RegisterRepresentation::Tagged(), "%context");
    instance.Asm().Parameter(0, RegisterRepresentation::Tagged(), "%this");
    for (int i = 0; i < parameter_count; ++i) {
      instance.parameters_.push_back(
          instance.Asm().Parameter(1 + i, RegisterRepresentation::Tagged()));
    }
    builder(instance);
    return instance;
  }

  Assembler& Asm() { return assembler_; }
  Graph& graph() { return *graph_; }
  Assembler& operator()() { return Asm(); }

  template <template <typename> typename... Reducers>
  void Run(bool trace_reductions = false) {
    TSAssembler<GraphVisitor, Reducers...> phase(
        graph(), graph().GetOrCreateCompanion(), zone_);
#ifdef DEBUG
    if (trace_reductions) {
      phase.template VisitGraph<true>();
    } else {
      phase.template VisitGraph<false>();
    }
#else
    phase.template VisitGraph<false>();
#endif
    // Map all captured inputs.
    for (auto& [key, captured] : captured_operations_) {
      std::set<OpIndex> temp = std::move(captured.generated_output);
      for (OpIndex index : graph_->AllOperationIndices()) {
        OpIndex origin = graph_->operation_origins()[index];
        if (temp.contains(origin)) captured.generated_output.insert(index);
      }
    }
  }

  V<Object> GetParameter(int index) {
    DCHECK_LE(0, index);
    DCHECK_LT(index, parameters_.size());
    return parameters_[index];
  }
  OpIndex BuildFrameState() {
    FrameStateData::Builder builder;
    // Closure
    builder.AddInput(MachineType::AnyTagged(),
                     Asm().SmiConstant(Smi::FromInt(0)));
    // TODO(nicohartmann@): Parameters, Context, Locals, Accumulator if
    // necessary.

    FrameStateFunctionInfo* function_info =
        zone_->template New<FrameStateFunctionInfo>(
            FrameStateType::kUnoptimizedFunction, 0, 0,
            Handle<SharedFunctionInfo>{});
    const FrameStateInfo* frame_state_info =
        zone_->template New<FrameStateInfo>(BytecodeOffset(0),
                                            OutputFrameStateCombine::Ignore(),
                                            function_info);

    return Asm().FrameState(
        builder.Inputs(), builder.inlined(),
        builder.AllocateFrameStateData(*frame_state_info, zone_));
  }

  OpIndex Capture(OpIndex input, const std::string& key) {
    captured_operations_[key] =
        CapturedOperation{this, input, std::set<OpIndex>{input}};
    return input;
  }
  template <typename T>
  V<T> Capture(V<T> input, const std::string& key) {
    return V<T>::Cast(Capture(static_cast<OpIndex>(input), key));
  }
  const CapturedOperation& GetCapture(const std::string& key) {
    auto it = captured_operations_.find(key);
    DCHECK_NE(it, captured_operations_.end());
    return it->second;
  }

  void PrintGraphForTurbolizer(const char* phase_name,
                               const char* suffix = "") {
    if (!stream_) {
      const testing::TestInfo* test_info =
          testing::UnitTest::GetInstance()->current_test_info();
      std::stringstream file_name;
      file_name << "turbo-" << test_info->test_suite_name() << "_"
                << test_info->name() << ".json";
      stream_ = std::make_unique<std::ofstream>(file_name.str(),
                                                std::ios_base::trunc);
      *stream_ << "{\"function\" : ";
      auto name =
          std::make_unique<char[]>(strlen("test_generated_function") + 1);
      strcpy(name.get(), "test_generated_function");
      JsonPrintFunctionSource(*stream_, -1, std::move(name), Handle<Script>{},
                              isolate_, Handle<SharedFunctionInfo>{});
      *stream_ << ",\n\"phases\":[";
    }
    PrintTurboshaftGraphForTurbolizer(*stream_, graph(), phase_name, nullptr,
                                      zone_);
  }

 private:
  TestInstance(std::unique_ptr<Graph> graph, Isolate* isolate, Zone* zone)
      : assembler_(*graph, *graph, zone),
        graph_(std::move(graph)),
        isolate_(isolate),
        zone_(zone) {}

  TSAssembler<> assembler_;
  std::unique_ptr<Graph> graph_;
  std::unique_ptr<std::ofstream> stream_;
  Isolate* isolate_;
  Zone* zone_;
  base::SmallMap<std::map<std::string, CapturedOperation>> captured_operations_;
  base::SmallVector<OpIndex, 4> parameters_;
};

class ReducerTest : public TestWithNativeContextAndZone {
 public:
  template <typename Builder>
  TestInstance CreateFromGraph(int parameter_count, const Builder& builder) {
    return TestInstance::CreateFromGraph(parameter_count, builder, isolate(),
                                         zone());
  }

  void SetUp() override {
    pipeline_data_.emplace(TurboshaftPipelineKind::kJS, info_, schedule_,
                           graph_zone_, this->zone(), broker_, isolate_,
                           source_positions_, node_origins_, sequence_, frame_,
                           assembler_options_, &max_unoptimized_frame_height_,
                           &max_pushed_argument_count_, instruction_zone_);
  }
  void TearDown() override { pipeline_data_.reset(); }

  // We use some dummy data to initialize the PipelineData::Scope.
  // TODO(nicohartmann@): Clean this up once PipelineData is reorganized.
  OptimizedCompilationInfo* info_ = nullptr;
  Schedule* schedule_ = nullptr;
  Zone* graph_zone_ = this->zone();
  JSHeapBroker* broker_ = nullptr;
  Isolate* isolate_ = this->isolate();
  SourcePositionTable* source_positions_ = nullptr;
  NodeOriginTable* node_origins_ = nullptr;
  InstructionSequence* sequence_ = nullptr;
  Frame* frame_ = nullptr;
  AssemblerOptions assembler_options_;
  size_t max_unoptimized_frame_height_ = 0;
  size_t max_pushed_argument_count_ = 0;
  Zone* instruction_zone_ = this->zone();

  base::Optional<turboshaft::PipelineData::Scope> pipeline_data_;
};

}  // namespace v8::internal::compiler::turboshaft
