#!/bin/bash

# Author: Sergio Durigan Junior <sergiodj@sergiodj.net>

# Disclaimer: This script contains bashisms.  It will probably not
# work very well with other shell interpreters.

# The source directory of selectize.js.  If building the Debian
# package, this is $(CURDIR)/../ (from debian/rules).
SOURCEDIR=

# The build directory, where we'll put the generated selectize.js.
BUILDDIR=

# The version of selectize.js being built.  'dpkg-parsechangelog -S
# Version' should be helpful here.
VERSION=


# Print the usage message on stdout.

usage ()
{
    cat <<EOF
$0 -- build selectize.js without using grant

Usage:

$0 SOURCE_DIR BUILD_DIR VERSION

Where:

  SOURCE_DIR -- The source directory (i.e., where selectize.js source
  is located).  If this script is being called from debian/rules, this
  is "\$(CURDIR)/../".

  BUILD_DIR -- Where the built files will be put.

  VERSION -- The version of selectize.js.
EOF
}

# Sanitize the arguments received via CLI.

sanitize_cli_args ()
{
    # source dir
    local sdir=$1
    # build dir
    local bdir=$2
    # version
    local ver=$3

    # Does the source directory exist?
    if test ! -d $sdir ; then
	echo "E: '$1' is not a valid directory."
	exit 1
    fi

    # Do we have a 'src/' directory inside the source directory?
    if test ! -d $sdir/src/ ; then
	echo "E: '$1/src/' doesn't exist.  Are you sure you provided the source directory?"
	exit 1
    fi

    # Change to the source directory.  That's where we'll make the
    # build.
    cd $sdir

    # Setting the global variables.
    SOURCEDIR=$sdir
    BUILDDIR=$bdir
    VERSION=$ver
}

# Prepare the build directory to receive the generated files.  This
# basically means removing the old directory and creating it again.

prepare_build_dir ()
{
    rm -rf $BUILDDIR
    mkdir -p $BUILDDIR
}

# Build selectize.js.
#
# This is the most important function of this script, of course.  It
# was heavily inspired by the original Gruntfile.js, from upstream.
# Since Debian does not ship Grunt, I had to write this "poor man's
# builder" for selectize.js.

build_js ()
{
    # List of files that will compose the final selectize.js script.
    # The order of the files on this list *matters*!
    local file_list="src/contrib/*.js src/constants.js src/utils.js \
src/selectize.js src/defaults.js src/selectize.jquery.js \
src/plugins/drag_drop/*.js src/plugins/dropdown_header/*.js \
src/plugins/optgroup_columns/*.js src/plugins/remove_button/*.js \
src/plugins/restore_on_backspace/*.js"
    # Temporary directory where we'll do some file manipulations.
    local tmpdir=`mktemp -d -p .`

    for files in $file_list ; do
	for file in `ls $files` ; do
	    # We'll do some modifications to the file, so we copy it
	    # to the temporary directory.
	    local b=`dirname $file`
	    mkdir -p $tmpdir/$b
	    cp -a $file $tmpdir/$file

	    if head -n1 $tmpdir/$file | grep -q '^\/\*\*' ; then
		# If the first line of the file is a comment, then
		# this means the file has a license header and we need
		# to remove it (Grunt calls it a 'banner').  The code
		# below is just a quick and dirty way of getting rid
		# of the banner without resorting to magical sed
		# expressions.
		local SAW_FIRST_COMMENT_BEGIN=0
		local SAW_FIRST_COMMENT_END=0

		while IFS= read -r line || test -n "$line" ; do
		    if test $SAW_FIRST_COMMENT_BEGIN -eq 0 ; then
			if echo "$line" | grep -q '^\/\*\*' ; then
			    SAW_FIRST_COMMENT_BEGIN=1
			    continue
			fi
		    elif test $SAW_FIRST_COMMENT_END -eq 0 ; then
			if echo "$line" | grep -q '^ \*\/$' ; then
			    SAW_FIRST_COMMENT_END=1
			fi
			continue
		    fi
		    echo "$line" >> $tmpdir/$file.1
		done < $tmpdir/$file

		mv $tmpdir/$file.1 $tmpdir/$file
	    fi

	    # Every line must be prepended with a TAB.
	    sed -i -e 's/^/\t/' $tmpdir/$file

	    # Concatenating the file.
	    cat $tmpdir/$file >> $BUILDDIR/selectize.js.1
	    /bin/echo -ne '\n\n\t\n' >> $BUILDDIR/selectize.js.1
	done
    done

    # Taking care of src/.wrapper.js now.
    cp -a src/.wrapper.js $tmpdir/
    sed -i "s/@@version/$VERSION/" $tmpdir/.wrapper.js

    local FOUND_JS=0
    # src/.wrapper.js is really a wrapper...  selectize.js.1 needs to
    # be put "in the middle" of it, and we need to get rid of a
    # special mark (@@js) where selectize.js.1 will be placed.
    while IFS= read -r line || test -n "$line" ; do
	if echo "$line" | grep -q '^\s@@js$' > /dev/null 2>&1 ; then
	    FOUND_JS=1
	    continue
	fi

	if test $FOUND_JS -eq 0 ; then
	    echo "$line" >> $BUILDDIR/selectize.js
	else
	    echo "$line" >> $BUILDDIR/selectize.js.epilogue
	fi
    done < $tmpdir/.wrapper.js

    # Final concatenation.
    cat $BUILDDIR/selectize.js.1 >> $BUILDDIR/selectize.js
    cat $BUILDDIR/selectize.js.epilogue >> $BUILDDIR/selectize.js

    # Cleaning up.
    rm -rf $tmpdir
}

# Check that the generated selectize.js is equivalent to the one that
# comes with the upstream source.  Note: this function will have to be
# rewritten/adjusted/skipped if we ever have to patch the JavaScript
# sources.

check_js ()
{
    if ! diff --unified \
	 --ignore-tab-expansion \
	 --ignore-space-change \
	 --ignore-blank-lines \
	 dist/js/selectize.js \
	 $BUILDDIR/selectize.js ; then
	echo "E: The generated selectize.js is NOT equal to the dist/js/selectize.js."
	exit 1
    fi
}

# Check that we received the exact number of arguments expected.  See
# 'usage' for an explanation of each argument.
if test $# -ne 3 ; then
    usage
    exit 1
fi

# Sanitize the arguments received.
sanitize_cli_args $@

# Prepare the build directory.
prepare_build_dir

# Build and check the generated JavaScript file.
build_js
check_js

exit 0
