package main

import (
	"context"
	"flag"
	"fmt"
	"io"
	"pkg.deepin.io/daemon/sync/infrastructure/storage"
	"strconv"
	"time"
)

var (
	token = flag.String("t", "", "the storage token")

	strg *storage.Storage

	duration = time.Second * 30
)

func main() {
	flag.Parse()
	if len(*token) == 0 {
		flag.Usage()
	}

	var err error
	strg, err = storage.NewStorage(*token)
	if err != nil {
		fmt.Println("Failed to new storage:", err)
		return
	}

	olist := doList()
	fmt.Println("List len:", len(olist))
	dlist := filterByDays(olist, 10)
	dumpList(dlist)
	nlist := filterByNum(olist, 10)
	dumpList(nlist)
}

func dumpList(list []*storage.ListObject) {
	fmt.Println("Dump list objs:")
	for _, obj := range list {
		fmt.Println("\tItem:", obj.Key, obj.Version, obj.Size)
	}
	fmt.Println("========")
}

func doList() []*storage.ListObject {
	ctx, cancel := context.WithTimeout(context.Background(), duration)
	list, err := strg.List(ctx, &storage.ListOptions{Prefix: "index"})
	cancel()
	if err != nil {
		fmt.Println("Failed to list index:", err)
		return nil
	}

	var objs []*storage.ListObject
	for {
		ctx, cancel := context.WithTimeout(context.Background(), duration)
		obj, err := list.Next(ctx)
		cancel()
		if err != nil {
			if err == io.EOF {
				break
			}
			fmt.Println("Failed to next:", err)
			break
		}
		objs = append(objs, obj)
	}
	return objs
}

func filterByDays(objs []*storage.ListObject, days int64) []*storage.ListObject {
	var ret []*storage.ListObject
	seconds := days * 24 * 60 * 60
	for _, obj := range objs {
		buf := []byte(obj.Version)
		ts := string(buf[:len(buf)-4])
		stamp, err := strconv.ParseInt(ts, 10, 64)
		if err != nil {
			fmt.Println("Failed to convert to timestamp:", obj.Key, obj.Version, ts)
			continue
		}
		if time.Now().Unix()-seconds > stamp {
			continue
		}
		ret = append(ret, obj)
	}
	return ret
}

func filterByNum(objs []*storage.ListObject, num int) []*storage.ListObject {
	if len(objs) < num {
		return objs
	}
	return objs[0:num]
}
