package helper

import (
	"fmt"
	"os"

	"github.com/godbus/dbus"
	hdmi "github.com/jouyouyun/hardware/dmi"
	"pkg.deepin.io/daemon/sync/infrastructure/hardware"
	. "pkg.deepin.io/daemon/sync/infrastructure/log"
)

const (
	helperDBusName = "com.deepin.sync.Helper"
	helperDBusPath = "/com/deepin/sync/Helper"
	helperDBusIFC  = helperDBusName
)

var (
	conn *dbus.Conn = nil

	ErrUninitialized = fmt.Errorf("uninitialized connection")
)

func Init() error {
	if conn != nil {
		return nil
	}

	var err error
	conn, err = dbus.SystemBus()
	if err != nil {
		return err
	}
	return nil
}

func GetHardware() (*hardware.Hardware, error) {
	if conn == nil {
		return nil, ErrUninitialized
	}
	var hw = hardware.Hardware{DMI: hdmi.DMI{}}
	err := helperGetObject().Call(helperDBusIFC+".GetHardware", 0).Store(&hw)
	if err != nil {
		return nil, err
	}
	hw.Username = os.Getenv("USER")
	return &hw, nil
}

// ConfirmPrivacy
func ConfirmPrivacy(id string) (error) {
	if conn == nil {
		return ErrUninitialized
	}
	err := helperGetObject().Call(helperDBusIFC+".ConfirmPrivacy", 0, id).Err
	return err
}

// HasConfirmPrivacy
func HasConfirmPrivacy(id string) (bool, error) {
	var hasConfirmPrivacy = false
	if conn == nil {
		return hasConfirmPrivacy, ErrUninitialized
	}
	err := helperGetObject().Call(helperDBusIFC+".HasConfirmPrivacy", 0, id).Store(&hasConfirmPrivacy)
	if err != nil {
		return hasConfirmPrivacy, err
	}
	return hasConfirmPrivacy, nil
}

func ConnectChanged(cb func(name string)) error {
	if conn == nil {
		return ErrUninitialized
	}
	var rule = fmt.Sprintf("type='signal',member='Changed',path='%s',interface='%s',sender='%s'",
		helperDBusPath, helperDBusIFC, helperDBusName)
	err := conn.BusObject().Call("org.freedesktop.DBus.AddMatch", 0, rule).Err
	if err != nil {
		Logger.Warning("Failed to watch bus rule:", err, rule)
		return err
	}
	var sig = make(chan *dbus.Signal, 1)
	conn.Signal(sig)
	for {
		c, ok := <-sig
		if !ok {
			Logger.Warning("Failed to recieve bus signal")
			break
		}
		fmt.Println("Signal:", c.Name, c.Path, c.Body)
		if c.Name != helperDBusIFC+".Changed" {
			continue
		}
		fmt.Println(len(c.Body))
		if len(c.Body) != 1 {
			continue
		}
		cb(c.Body[0].(string))
	}
	return nil
}

func helperGetObject() dbus.BusObject {
	return conn.Object(helperDBusName, helperDBusPath)
}
