package deepinid

import (
	"time"

	"pkg.deepin.io/daemon/sync/infrastructure/utils"

	"encoding/gob"
	"os"
	"path/filepath"
	"sync"
)

const (
	tokenFilePath = ".config/deepin/deepin-sync-daemon/token"
)

type tokenData struct {
	IsLoggedIn  bool   `json:"is_logged_in"`
	UserID      int64  `json:"user_id"`
	Username    string `json:"username"`
	Region      string `json:"region"`
	Nickname    string `json:"nickname"`
	Token       string `json:"token"`
	AccessToken string `json:"access_token"`
	Expiry      int64  `json:"expiry"`
	HardwareID  string `json:"hardware_id"`
}

var (
	_tokenLocker sync.RWMutex
)

func (td *tokenData) isExpired() bool {
	return (td.Expiry - 1*60) < time.Now().Unix()
}

func saveToken(td *tokenData) error {
	_tokenLocker.Lock()
	defer _tokenLocker.Unlock()
	filename := filepath.Join(os.Getenv("HOME"), tokenFilePath)
	err := utils.EnsureParentDirExists(filename)
	if err != nil {
		return err
	}

	fw, err := os.OpenFile(filename, os.O_CREATE|os.O_TRUNC|os.O_WRONLY, 0644)
	if err != nil {
		return err
	}
	defer fw.Close()

	var enc = gob.NewEncoder(fw)
	return enc.Encode(td)
}

func readToken() (*tokenData, error) {
	_tokenLocker.RLock()
	defer _tokenLocker.RUnlock()
	filename := filepath.Join(os.Getenv("HOME"), tokenFilePath)
	fr, err := os.OpenFile(filename, os.O_RDONLY, 0644)
	if err != nil {
		return nil, err
	}
	defer fr.Close()

	var dec = gob.NewDecoder(fr)
	var td tokenData
	err = dec.Decode(&td)
	if err != nil {
		return nil, err
	}
	return &td, nil
}

func removeToken() error {
	_tokenLocker.Lock()
	defer _tokenLocker.Unlock()
	filename := filepath.Join(os.Getenv("HOME"), tokenFilePath)
	return os.RemoveAll(filename)
}
