// SPDX-FileCopyrightText: 2018 - 2023 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include "networkhelper.h"

#include <NetworkManagerQt/WirelessDevice>
#include <NetworkManagerQt/Manager>

using namespace dde::network;

NetworkHelper::NetworkHelper(QObject *parent)
    : QObject(parent)
{
    NetworkManager::Device::List devices = NetworkManager::networkInterfaces();
    for (NetworkManager::Device::Ptr device : devices) {
        if (device->type() != NetworkManager::Device::Type::Wifi)
            continue;

        NetworkManager::WirelessDevice::Ptr wirelessDevice = device.staticCast<NetworkManager::WirelessDevice>();
        connect(wirelessDevice.data(), &NetworkManager::WirelessDevice::accessPointAppeared, this, &NetworkHelper::onAccessPointAppeared);
        connect(wirelessDevice.data(), &NetworkManager::WirelessDevice::accessPointDisappeared, this, &NetworkHelper::onAccessPointDisappeared);
        QStringList apPaths = wirelessDevice->accessPoints();
        QJsonArray array;
        for (const QString &path : apPaths) {
            QJsonObject json = makeJsonObject(wirelessDevice->findAccessPoint(path));
            if (json.isEmpty())
                continue;

            array.append(json);
        }
        m_json.insert(wirelessDevice->uni(), array);
    }
}

NetworkHelper::~NetworkHelper()
{
}

QJsonObject NetworkHelper::accessPointJson() const
{
    return m_json;
}

QJsonObject NetworkHelper::makeJsonObject(const NetworkManager::AccessPoint::Ptr &accessPoint)
{
    if (accessPoint.isNull() || accessPoint->ssid().isEmpty())
        return QJsonObject();

    QJsonObject apJson;
    apJson.insert("Ssid", accessPoint->ssid());
    bool secured = (accessPoint->capabilities() == NetworkManager::AccessPoint::Capability::Privacy || accessPoint->wpaFlags() != 0 || accessPoint->rsnFlags() != 0);
    apJson.insert("Secured", secured);
    apJson.insert("SecuredInEap", false);
    apJson.insert("Strength", accessPoint->signalStrength());
    apJson.insert("Path", accessPoint->uni());
    apJson.insert("Frequency", static_cast<int>(accessPoint->frequency()));
    apJson.insert("Hidden", false);
    apJson.insert("Flags", static_cast<int>(accessPoint->capabilities()));
    return apJson;
}

void NetworkHelper::onAccessPointAppeared(const QString &uni)
{
    NetworkManager::WirelessDevice *device = static_cast<NetworkManager::WirelessDevice *>(sender());
    QJsonObject json = makeJsonObject(device->findAccessPoint(uni));
    if (json.isEmpty())
        return;

    QJsonArray array = m_json[device->uni()].toArray();
    array.append(json);
    m_json.insert(device->uni(), array);
    emit accessPointChanged(m_json);
}

void NetworkHelper::onAccessPointDisappeared(const QString &uni)
{
    NetworkManager::WirelessDevice *device = static_cast<NetworkManager::WirelessDevice *>(sender());
    QJsonArray array = m_json[device->uni()].toArray();
    for (int i = array.size() - 1; i >= 0; i--) {
        QJsonValue jsonValue = array[i];
        if (!jsonValue.isObject())
            continue;

        QJsonObject jsonObject = jsonValue.toObject();
        if (jsonObject.value("Path").toString() != uni)
            continue;

        array.removeAt(i);
    }
    m_json.insert(device->uni(), array);
    emit accessPointChanged(m_json);
}
