// SPDX-FileCopyrightText: 2018 - 2022 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include "devicemanagerrealize.h"
#include "dslcontrollernm.h"
#include "hotspotcontrollernm.h"
#include "networkdevicebase.h"
#include "networkmanagerprocesser.h"
#include "proxycontroller.h"
#include "vpncontrollernm.h"
#include "ipconfilctchecker.h"
#include "networkdetails.h"
#include "netutils.h"

#include <NetworkManagerQt/WiredDevice>
#include <NetworkManagerQt/WirelessDevice>
#include <NetworkManagerQt/Manager>
#include <NetworkManagerQt/Settings>
#include <NetworkManagerQt/Security8021xSetting>
#include <NetworkManagerQt/IpAddress>
#include <NetworkManagerQt/Ipv4Setting>
#include <NetworkManagerQt/Ipv6Setting>
#include <NetworkManagerQt/WirelessSetting>

#include <wireddevice.h>
#include <wirelessdevice.h>

using namespace dde::network;

const static QString NetworkManagerService = "org.freedesktop.NetworkManager";
const static QString NetworkManagerPath = "/org/freedesktop/NetworkManager";
const static QString NetworkManagerInterface = "org.freedesktop.NetworkManager";

NetworkManagerProcesser::NetworkManagerProcesser(bool ipCheck, QObject *parent)
    : NetworkProcesser(parent)
    , ProcesserInterface()
    , m_proxyController(Q_NULLPTR)
    , m_vpnController(Q_NULLPTR)
    , m_dslController(Q_NULLPTR)
    , m_hotspotController(Q_NULLPTR)
    , m_connectivity(dde::network::Connectivity::Unknownconnectivity)
    , m_ipChecker(new IPConfilctChecker(this, ipCheck))
    , m_needDetails(false)
{
    QDBusMessage getDevices = QDBusMessage::createMethodCall(NetworkManagerService, NetworkManagerPath, NetworkManagerInterface, "GetAllDevices");
    QDBusConnection::systemBus().callWithCallback(getDevices, this, SLOT(onDevicesChanged(QList<QDBusObjectPath>)));
    initConnections();
    onConnectivityChanged(NetworkManager::connectivity());
    QDBusMessage checkConnectivity = QDBusMessage::createMethodCall(NetworkManagerService, NetworkManagerPath, NetworkManagerInterface, "CheckConnectivity");
    QDBusConnection::systemBus().callWithCallback(checkConnectivity, this, SLOT(checkConnectivityFinished(quint32)));
}

NetworkManagerProcesser::~NetworkManagerProcesser()
{
    delete m_ipChecker;
}

void NetworkManagerProcesser::initConnections()
{
    connect(NetworkManager::notifier(), &NetworkManager::Notifier::deviceAdded, this, [ this ](const QString &uni) {
        qInfo() << "device added:" << uni;
        onDeviceAdded(uni);
    });
    connect(NetworkManager::notifier(), &NetworkManager::Notifier::deviceRemoved, this, [ this ](const QString &uni) {
        qInfo() << "device removed:" << uni;
        onDeviceRemove(uni);
        if (m_hotspotController) {
            m_hotspotController->updateDevices(m_devices);
        }
    });
    connect(NetworkManager::notifier(), &NetworkManager::Notifier::connectivityChanged, this, &NetworkManagerProcesser::onConnectivityChanged);
}

void NetworkManagerProcesser::onDevicesChanged(const QList<QDBusObjectPath> &devices)
{
    for(QDBusObjectPath device : devices ) {
        qInfo() << "Device added: " << device.path();
        onDeviceAdded(device.path());
    }
}

QList<NetworkDeviceBase *> NetworkManagerProcesser::devices()
{
    return m_devices;
}

dde::network::Connectivity NetworkManagerProcesser::connectivity()
{
    return m_connectivity;
}

QList<NetworkDetails *> NetworkManagerProcesser::networkDetails()
{
    if (!m_needDetails) {
        m_needDetails = true;
        updateNetworkDetail();
    }

    return m_details;
}

ProxyController *NetworkManagerProcesser::proxyController()
{
    if (!m_proxyController)
        m_proxyController = new ProxyController(this);

    return m_proxyController;
}

VPNController *NetworkManagerProcesser::vpnController()
{
    if (!m_vpnController)
        m_vpnController = new VPNController_NM(this);

    return m_vpnController;
}

DSLController *NetworkManagerProcesser::dslController()
{
    if (!m_dslController)
        m_dslController = new DSLController_NM(this);

    return m_dslController;
}

HotspotController *NetworkManagerProcesser::hotspotController()
{
    if (!m_hotspotController) {
        m_hotspotController = new HotspotController_NM(this);
        m_hotspotController->updateDevices(m_devices);
    }

    return m_hotspotController;
}

void NetworkManagerProcesser::deviceEnabledChanged()
{
    if (m_hotspotController)
        m_hotspotController->updateDevices(m_devices);
}

void NetworkManagerProcesser::sortDevice()
{
    auto getPathIndex = [](const QString path)->int {
        int index = path.lastIndexOf("/");
        QString tmpIndexValue = path.mid(index + 1);
        return tmpIndexValue.toInt();
    };
    // 有线网络始终在无线网络的前面，如果两者都是有线或者无线网络，则按照path的顺序来排序
    qSort(m_devices.begin(), m_devices.end(),  [ = ](NetworkDeviceBase *device1, NetworkDeviceBase *device2) {
        if (device1->deviceType() == DeviceType::Wired && device2->deviceType() == DeviceType::Wireless)
            return true;

        if (device1->deviceType() == DeviceType::Wireless && device2->deviceType() == DeviceType::Wired)
            return false;

        return getPathIndex(device1->path()) < getPathIndex(device2->path());
    });
}

void NetworkManagerProcesser::updateNetworkDetail()
{
    if (!m_needDetails)
        return;

    for (NetworkDetails *networkDetail : m_details)
        delete networkDetail;

    m_details.clear();
    QJsonArray jsonArray = networkDetail();
    for (const QJsonValue json : jsonArray) {
        QJsonObject jsonObject = json.toObject();
        NetworkDetails *detail = new NetworkDetails(this);
        detail->updateData(jsonObject);
        m_details << detail;
    }
    Q_EMIT activeConnectionChange();
}

static QString getSecurity(NetworkManager::WirelessDevice::Ptr wirelessDevice)
{
    if (wirelessDevice.isNull())
        return QObject::tr("None");

    NetworkManager::AccessPoint::Ptr activeAccessPoint = wirelessDevice->activeAccessPoint();
    NetworkManager::ActiveConnection::Ptr activeConnection = wirelessDevice->activeConnection();
    if (activeConnection.isNull() || activeAccessPoint.isNull())
        return QString();

    NetworkManager::ConnectionSettings::ConnectionType connectionType = activeConnection->connection()->settings()->connectionType();
    if (connectionType != NetworkManager::ConnectionSettings::ConnectionType::Wireless || activeAccessPoint.isNull())
        return QString();

    NetworkManager::WirelessSecuritySetting::KeyMgmt keyMemt;
    NetworkManager::WirelessSecuritySetting::Ptr securitySetting = activeConnection->connection()->settings()->setting(
                NetworkManager::Setting::SettingType::WirelessSecurity).dynamicCast<NetworkManager::WirelessSecuritySetting>();
    if (securitySetting.isNull()) {
        keyMemt = WirelessDeviceManagerRealize::getKeyMgmtByAp(activeAccessPoint->uni());
    } else {
        keyMemt = securitySetting->keyMgmt();
    }
    QString security = QObject::tr("None");
    switch (keyMemt) {
    case NetworkManager::WirelessSecuritySetting::KeyMgmt::Wep:
        security = QObject::tr("WEP 40/128-bit Key");
        break;
    case NetworkManager::WirelessSecuritySetting::KeyMgmt::WpaPsk:
        security = QObject::tr("WPA/WPA2 Personal");
        break;
    case NetworkManager::WirelessSecuritySetting::KeyMgmt::WpaSae:
        security = QObject::tr("WPA3 Personal");
        break;
    case NetworkManager::WirelessSecuritySetting::KeyMgmt::WpaEap: {
        NetworkManager::Security8021xSetting::Ptr wirelessSecuritySetting = activeConnection->connection()->settings()->setting(NetworkManager::Setting::SettingType::Security8021x).dynamicCast<NetworkManager::Security8021xSetting>();
        if (!wirelessSecuritySetting.isNull()) {
            const QList<NetworkManager::Security8021xSetting::EapMethod> &eapMethods = wirelessSecuritySetting->eapMethods();
            NetworkManager::Security8021xSetting::EapMethod currentEapMethod = eapMethods.isEmpty() ? NetworkManager::Security8021xSetting::EapMethodTls : eapMethods.first();
            switch (currentEapMethod) {
            case NetworkManager::Security8021xSetting::EapMethodTls:
                security = "EAP" + QObject::tr("TLS");
                break;
            case NetworkManager::Security8021xSetting::EapMethodMd5:
                security = "EAP/" + QObject::tr("MD5");
                break;
            case NetworkManager::Security8021xSetting::EapMethodLeap:
                security = "EAP/" + QObject::tr("LEAP");
                break;
            case NetworkManager::Security8021xSetting::EapMethodFast:
                security = "EAP/" + QObject::tr("FAST");
                break;
            case NetworkManager::Security8021xSetting::EapMethodTtls:
                security = "EAP/" + QObject::tr("Tunneled TLS");
                break;
            case NetworkManager::Security8021xSetting::EapMethodPeap:
                security = "EAP/" + QObject::tr("Protected EAP");
                break;
            default:
                break;
            }
        }
        break;
    }
    default:
        break;
    }
    return security;
}

QJsonArray NetworkManagerProcesser::networkDetail() const
{
    QJsonArray array;
    NetworkManager::Device::List allDevices = NetworkManager::networkInterfaces();
    for (NetworkManager::Device::Ptr device : allDevices) {
        // 如果当前未连接状态，则无需显示
        NetworkManager::ActiveConnection::Ptr activeConnection = device->activeConnection();
        if (activeConnection.isNull() || device->state() != NetworkManager::Device::State::Activated)
            continue;

        bool isHotspot = false;
        QJsonObject json;
        switch (device->type()) {
        case NetworkManager::Device::Type::Wifi: {
            NetworkManager::WirelessDevice::Ptr wirelessDevice = device.staticCast<NetworkManager::WirelessDevice>();
            NetworkManager::ConnectionSettings::ConnectionType connectionType = activeConnection->connection()->settings()->connectionType();
            if (wirelessDevice->mode() == NetworkManager::WirelessDevice::OperationMode::ApMode) {
                // 开启了热点
                json.insert("ConnectionName", tr("Hotspot"));
                json.insert("ConnectionType", "wireless-hotspot");
                isHotspot = true;
            } else {
                // 未开启热点的情况下，需要显示协议等数据
                NetworkManager::WirelessSetting::Ptr wirelessSetting = activeConnection->connection()->settings()->setting(NetworkManager::Setting::SettingType::Wireless).dynamicCast<NetworkManager::WirelessSetting>();
                if (!wirelessSetting.isNull())
                    json.insert("ConnectionName", QString(wirelessSetting->ssid()));

                json.insert("ConnectionType", "wireless");
                // 获取协议信息
                json.insert("Protocol", NetworkManager::ConnectionSettings::typeAsString(connectionType));
            }
            json.insert("Security", getSecurity(wirelessDevice));
            NetworkManager::AccessPoint::Ptr activeAccessPoint = wirelessDevice->activeAccessPoint();
            if (!activeAccessPoint.isNull()) {
                QJsonObject hotspotInfo;
                hotspotInfo.insert("Ssid", activeAccessPoint->ssid());
                uint frequency = activeAccessPoint->frequency();
                QString band;
                if (frequency >= 4915 && frequency <= 5825)
                    band = "a";
                else if (frequency >= 2412 && frequency <= 2484)
                    band = "bg";
                else
                    band = "unknown";
                hotspotInfo.insert("Band", band);
                static QMap<uint, int> frequencyChannelMap {
                    {2412, 1}, {2417, 2}, {2422, 3}, {2427, 4}, {2432, 5}, {2437, 6}, {2442, 7},
                    {2447, 8}, {2452, 9}, {2457, 10}, {2462, 11}, {2467, 12}, {2472, 13}, {2484, 14},
                    {5035, 7}, {5040, 8}, {5045, 9}, {5055, 11}, {5060, 12}, {5080, 16}, {5170, 34},
                    {5180, 36}, {5190, 38}, {5200, 40}, {5220, 44}, {5230, 44}, {5240, 48}, {5260, 52}, {5280, 56}, {5300, 60},
                    {5320, 64}, {5500, 100}, {5520, 104}, {5540, 108}, {5560, 112}, {5580, 116}, {5600, 120},
                    {5620, 124}, {5640, 128}, {5660, 132}, {5680, 136}, {5700, 140}, {5745, 149}, {5765, 153},
                    {5785, 157}, {5805, 161}, {5825, 165}, {4915, 183}, {4920, 184}, {4925, 185}, {4935, 187},
                    {4940, 188}, {4945, 189}, {4960, 192}, {4980, 196}};
                hotspotInfo.insert("Channel", frequencyChannelMap.value(frequency, 0));
                json.insert("Hotspot", hotspotInfo);
            }
            json.insert("HwAddress", wirelessDevice->hardwareAddress());
            uint speed = static_cast<uint>(wirelessDevice->bitRate() / 1000);
            json.insert("Speed", QString("%1 Mbps").arg(speed));
        }
            break;
        case NetworkManager::Device::Type::Ethernet: {
            NetworkManager::WiredDevice::Ptr wiredDevice = device.staticCast<NetworkManager::WiredDevice>();
            NetworkManager::ActiveConnection::Ptr activeConnection = wiredDevice->activeConnection();
            if (!activeConnection.isNull())
                json.insert("ConnectionName", activeConnection->id());
            json.insert("HwAddress", wiredDevice->hardwareAddress());
            uint speed = static_cast<uint>(wiredDevice->bitRate() / 1000);
            json.insert("Speed", QString("%1 Mbps").arg(speed));
        }
            break;
        default:
            continue;
        }
        json.insert("Device", device->uni());
        json.insert("DeviceInterface", device->interfaceName());
        if (!isHotspot) {
            // 获取IPV4的地址
            QJsonObject ipV4Json;
            QJsonArray ipV4Array;
            NetworkManager::IpConfig ipV4Config = device->ipV4Config();
            QList<NetworkManager::IpAddress> addresses = ipV4Config.addresses();
            for (NetworkManager::IpAddress address : addresses) {
                QJsonObject ipAddr;
                ipAddr.insert("Address", address.ip().toString());
                ipAddr.insert("Prefix", address.prefixLength());
                ipV4Array.append(ipAddr);
            }
            ipV4Json.insert("Addresses", ipV4Array);
            ipV4Json.insert("Gateway", ipV4Config.gateway());
            QJsonArray ipV4NameServersArray;
            QList<QHostAddress> ipV4NameServers = ipV4Config.nameservers();
            for (const QHostAddress &ipV4NameServer : ipV4NameServers) {
                ipV4NameServersArray.append(ipV4NameServer.toString());
            }
            ipV4Json.insert("Nameservers", ipV4NameServersArray);
            json.insert("IPv4", ipV4Json);

            // 获取IpV6的地址
            QJsonObject ipV6Json;
            QJsonArray ipV6Array;
            NetworkManager::IpConfig ipV6Config = device->ipV6Config();
            QList<NetworkManager::IpAddress> ipV6addresses = ipV6Config.addresses();
            if (ipV6addresses.size() > 0) {
                for (NetworkManager::IpAddress address : ipV6addresses) {
                    QJsonObject ipAddr;
                    ipAddr.insert("Address", address.ip().toString());
                    ipAddr.insert("Prefix", address.prefixLength());
                    ipV6Array.append(ipAddr);
                }
            } else {
                QJsonObject ipAddr;
                ipAddr.insert("Address", "0::0");
                ipAddr.insert("Prefix", "0");
                ipV6Array.append(ipAddr);
            }
            ipV6Json.insert("Addresses", ipV6Array);
            ipV6Json.insert("Gateway", ipV6Config.gateway());
            QJsonArray ipV6NameServersArray;
            QList<QHostAddress> ipV6NameServers = ipV6Config.nameservers();
            for (const QHostAddress &nameServer : ipV6NameServers) {
                ipV6NameServersArray.append(nameServer.toString());
            }
            ipV6Json.insert("Nameservers", ipV6NameServersArray);
            json.insert("IPv6", ipV6Json);
        }
        array.append(json);
    }

    return array;
}

void NetworkManagerProcesser::onDeviceAdded(const QString &uni)
{
    auto deviceExist = [ this ] (const QString &uni)->bool {
        for (NetworkDeviceBase *device : m_devices) {
            if (device->path() == uni)
                return true;
        }

        return false;
    };

    if (deviceExist(uni))
        return;

    NetworkManager::Device::Ptr currentDevice(nullptr);
    NetworkManager::Device::List allDevices = NetworkManager::networkInterfaces();
    for (NetworkManager::Device::Ptr device : allDevices) {
        if (device->uni() != uni)
            continue;

        if (device->type() == NetworkManager::Device::Wifi || device->type() == NetworkManager::Device::Ethernet)
            currentDevice = device;

        break;
    }

    if (!currentDevice)
        return;

    auto updateHotspot = [ this ] {
        if (m_hotspotController) {
            m_hotspotController->updateDevices(m_devices);
        }
    };

    auto createDevice = [ = ](const NetworkManager::Device::Ptr &device)->NetworkDeviceBase * {
        if (device->type() == NetworkManager::Device::Wifi) {
            // 无线网络
            NetworkManager::WirelessDevice::Ptr wDevice = device.staticCast<NetworkManager::WirelessDevice>();
            WirelessDeviceManagerRealize *deviceRealize = new WirelessDeviceManagerRealize(m_ipChecker, wDevice);
            deviceRealize->setConnectivity(m_connectivity);
            deviceRealize->addProcesser(this);
            return new WirelessDevice(deviceRealize, this);
        }

        if (device->type() == NetworkManager::Device::Ethernet) {
            // 有线网络
            NetworkManager::WiredDevice::Ptr wDevice = device.staticCast<NetworkManager::WiredDevice>();
            DeviceManagerRealize *deviceRealize = new WiredDeviceManagerRealize(m_ipChecker, wDevice);
            deviceRealize->setConnectivity(m_connectivity);
            return new WiredDevice(deviceRealize, this);
        }

        return nullptr;
    };

    // 无线网卡不管是否down，都显示，因为在开启飞行模式后，需要显示网卡的信息
    auto deviceCreateOrRemove = [ this, deviceExist, createDevice, updateHotspot ](const NetworkManager::Device::Ptr &device) {
        if (device->managed() && (
#ifdef USE_DEEPIN_NMQT
                    (device->interfaceFlags() & DEVICE_INTERFACE_FLAG_UP) ||
#endif
                    device->type() == NetworkManager::Device::Wifi)) {
            // 如果由非manager变成manager的模式，则新增设备
            if (!deviceExist(device->uni())) {
                NetworkDeviceBase *newDevice = createDevice(device);
                if (newDevice) {
                    m_devices << newDevice;
                    sortDevice();
                    updateDeviceName();
                    updateNetworkDetail();
                    Q_EMIT deviceAdded({ newDevice });
                    updateHotspot();
                }
            }
        } else {
            // 如果由manager变成非manager模式，则移除设备
            NetworkDeviceBase *rmDevice = nullptr;
            for (NetworkDeviceBase *dev : m_devices) {
                if (dev->path() == device->uni()) {
                    m_devices.removeOne(dev);
                    rmDevice = dev;
                    break;
                }
            }
            if (rmDevice) {
                Q_EMIT rmDevice->removed();
                sortDevice();
                updateDeviceName();
                updateNetworkDetail();
                Q_EMIT deviceRemoved({ rmDevice });
                delete rmDevice;
                rmDevice = nullptr;
                updateHotspot();
            }
        }
    };

    NetworkDeviceBase *newDevice = Q_NULLPTR;
    if (currentDevice->managed() && (
#ifdef USE_DEEPIN_NMQT
                (currentDevice->interfaceFlags() & DEVICE_INTERFACE_FLAG_UP) ||
#endif
                currentDevice->type() == NetworkManager::Device::Wifi))
        newDevice = createDevice(currentDevice);
#ifdef USE_DEEPIN_NMQT
    connect(currentDevice.get(), &NetworkManager::Device::interfaceFlagsChanged, this, [ currentDevice, deviceCreateOrRemove ] {
        deviceCreateOrRemove(currentDevice);
    });
#endif
    connect(currentDevice.get(), &NetworkManager::Device::managedChanged, this, [ currentDevice, deviceCreateOrRemove ] {
        deviceCreateOrRemove(currentDevice);
    });

    if (newDevice) {
        connect(newDevice, &NetworkDeviceBase::deviceStatusChanged, this, &NetworkManagerProcesser::updateNetworkDetail);
        m_devices << newDevice;
        sortDevice();
        updateDeviceName();
        updateNetworkDetail();
        Q_EMIT deviceAdded({ newDevice });
        updateHotspot();
    }
}

void NetworkManagerProcesser::onDeviceRemove(const QString &uni)
{
    NetworkDeviceBase *rmDevice = Q_NULLPTR;
    for (NetworkDeviceBase *device : m_devices) {
        if (device->path() == uni) {
            m_devices.removeOne(device);
            rmDevice = device;
            break;
        }
    }

    if (rmDevice) {
        Q_EMIT rmDevice->removed();
        sortDevice();
        updateDeviceName();
        updateNetworkDetail();
        Q_EMIT deviceRemoved({ rmDevice });
        delete rmDevice;
    }
}

void NetworkManagerProcesser::checkConnectivityFinished(quint32 conntity)
{
    onConnectivityChanged(NetworkManager::Connectivity(conntity));
}

void NetworkManagerProcesser::onConnectivityChanged(NetworkManager::Connectivity conntity)
{
    dde::network::Connectivity ctity;
    switch (conntity) {
    case NetworkManager::Connectivity::Full: {
        ctity = dde::network::Connectivity::Full;
        break;
    }
    case NetworkManager::Connectivity::Portal: {
        ctity = dde::network::Connectivity::Portal;
        break;
    }
    case NetworkManager::Connectivity::Limited: {
        ctity = dde::network::Connectivity::Limited;
        break;
    }
    case NetworkManager::Connectivity::NoConnectivity: {
        ctity = dde::network::Connectivity::Noconnectivity;
        break;
    }
    case NetworkManager::Connectivity::UnknownConnectivity: {
        ctity = dde::network::Connectivity::Unknownconnectivity;
        break;
    }
    }
    if (m_connectivity != ctity) {
        m_connectivity = ctity;
        Q_EMIT connectivityChanged(m_connectivity);
    }
}

// 虚接口
ProcesserInterface::ProcesserInterface()
{
}

ProcesserInterface::~ProcesserInterface()
{
}
