// SPDX-FileCopyrightText: 2011 - 2022 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: LGPL-3.0-or-later

#include "commoninfowidget.h"
#include "window/insertplugin.h"
#include "window/utils.h"
#include "window/gsettingwatcher.h"
#include "widgets/multiselectlistview.h"


#include <DStandardItem>

#include <QVBoxLayout>
#include <QStandardItemModel>
#include <QList>
#include <QIcon>
#include <QDebug>

DWIDGET_USE_NAMESPACE

using namespace DCC_NAMESPACE;
using namespace commoninfo;
#define GSETTINGS_CONTENS_SERVER "iscontens-server"

CommonInfoWidget::CommonInfoWidget(QWidget *parent)
    : QWidget(parent)
    , m_vBoxLayout(new QVBoxLayout(this))
    , m_listView(new dcc::widgets::MultiSelectListView(this))
    , m_itemModel(new QStandardItemModel(this))
    , isContensServer(false)
{
    initWidget();
    initData();
}

CommonInfoWidget::~CommonInfoWidget()
{

}

dcc::widgets::MultiSelectListView *CommonInfoWidget::getCommonListView()
{
    return m_listView;
}

void CommonInfoWidget::showDefaultWidget()
{
    for (int i = 0; i < m_itemModel->rowCount(); i++) {
        if (!m_listView->isRowHidden(i)) {
            m_listView->activated(m_itemModel->index(i, 0));
            break;
        }
    }
}

bool CommonInfoWidget::configContent(const QString &configName)
{
    for (auto m : m_itemList) {
        if (configName == m.gsettingsName)
            return true;
    }
    return false;
}

void CommonInfoWidget::initWidget()
{
    setAccessibleName("CommonInfoWidget");
    m_listView->setAccessibleName("List_commonlist");
    m_listView->setViewportMargins(ScrollAreaMargins);
    m_listView->setFrameShape(QFrame::NoFrame);
    m_listView->setEditTriggers(QListView::NoEditTriggers);
    m_listView->setIconSize(ListViweIconSize);
    m_listView->setModel(m_itemModel);
    m_listView->setItemSize(ListViweSysItemSize);

    m_vBoxLayout->setContentsMargins(5, 5, 5, 0);
    m_vBoxLayout->setMargin(0);
    m_vBoxLayout->addWidget(m_listView);

    setLayout(m_vBoxLayout);
}

void CommonInfoWidget::initData()
{
    m_itemList.append({"dcc_boot_menu", tr("Boot Menu"),
                       QMetaMethod::fromSignal(&CommonInfoWidget::requestShowBootWidget), nullptr, "bootMenu"});

    //以下模块只在非服务器版本使用
    if (!IsServerSystem && !IsCommunitySystem) {
        if (DSysInfo::uosEditionType() != DSysInfo::UosEuler || DSysInfo::uosEditionType() != DSysInfo::UosEnterpriseC) {
            if (!DisableDeveloperMode) {
                m_itemList.append({ "dcc_developer_mode",
                                    tr("Developer Mode"),
                                    QMetaMethod::fromSignal(&CommonInfoWidget::requestShowDeveloperModeWidget),
                                    nullptr,
                                    "developerMode"
                                  });
            }
        }
        m_itemList.append({"dcc_ue_plan", tr("User Experience Program"),
                           QMetaMethod::fromSignal(&CommonInfoWidget::requestShowUEPlanWidget), nullptr, "userExperienceProgram"});

#if 0
        m_itemList.append({"", tr("Tablet Mode"),
                           QMetaMethod::fromSignal(&CommonInfoWidget::requestShowTabletModeWidget)
                          });
#endif
    }

    for (auto m : m_itemList) {
        DStandardItem *item = new DStandardItem;
        item->setIcon(QIcon::fromTheme(m.itemIcon));
        item->setText(m.itemText);
        m_itemModel->appendRow(item);
        GSettingWatcher::instance()->bind(m.gsettingsName, m_listView, item);
    }

    if (InsertPlugin::instance()->updatePluginInfo("commoninfo"))
        InsertPlugin::instance()->pushPlugin(m_itemModel, m_itemList);

    connect(m_listView, &DListView::clicked, this, [&](const QModelIndex & index) {
        if (m_lastIndex == index) return;

        m_lastIndex = index;

        m_itemList[index.row()].itemSignal.invoke(m_itemList[index.row()].plugin ? m_itemList[index.row()].plugin : this);
        m_listView->resetStatus(index);
    });
    connect(m_listView, &DListView::activated, m_listView, &QListView::clicked);
    connect(GSettingWatcher::instance(), &GSettingWatcher::requestUpdateSecondMenu, this, [=](int row, const QString & name) {
        //不是本模块配置不响应
        if (!configContent(name))
            return ;
        bool isAllHidden = true;
        for (int i = 0; i < m_itemModel->rowCount(); i++) {
            if (!m_listView->isRowHidden(i))
                isAllHidden = false;
        }

        if (m_listView->selectionModel()->selectedRows().size() > 0) {
            int index = m_listView->selectionModel()->selectedRows()[0].row();
            Q_EMIT requestUpdateSecondMenu(index == row);
        } else {
                Q_EMIT requestUpdateSecondMenu(false);
        }

        if (isAllHidden) {
            m_lastIndex = QModelIndex();
            m_listView->clearSelection();
        }
    });
}

int CommonInfoWidget::showPath(const QString& path)
{
    qDebug() << "show path:" << path;
    for (int i = 0; i < m_itemList.size(); ++i) {
        auto menu = m_itemList[i];
        if (tr(path.toStdString().c_str()) == menu.itemText) {
            setCurrentIndex(i);
            return 0;
        }
    }
    return -1;
}

void CommonInfoWidget::setCurrentIndex(const int index)
{
    QModelIndex mindex = m_itemModel->index(index, 0);
    m_listView->setCurrentIndex(mindex);
    Q_EMIT m_listView->clicked(mindex);
}
